/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.dam.scene7.api;

import java.io.InputStream;

import org.apache.sling.api.resource.ResourceResolver;
import org.w3c.dom.Document;

import aQute.bnd.annotation.ProviderType;

import com.day.cq.dam.scene7.api.constants.Scene7JobType;
import com.day.cq.dam.scene7.api.model.Scene7Asset;
import com.day.cq.dam.scene7.impl.MetadataCondition;

/**
 * The Scene7APIClient provides the Scene7 SPS API methods used by CQ DAM. For more details about the methods please consult the SPS API.
 */
@ProviderType
public interface Scene7APIClient {

    /**
     * default content-type (i.e. 'application/soap+xml'), defaults to 'text/xml'
     */
    static final String CONTENTTYPE = "text/xml";

    static final int RECORDS_PER_PAGE_NO_FILTER = 1000;
    static final int RECORDS_PER_PAGE_WITH_FILTER = 10000;

    /**
     * Deletes an asset from Scene7.
     * 
     * @param assetHandle
     *            the handle to the asset to delete
     * @param s7Config
     *            the Scene7 configuration
     * @return the API's response wrapped in a {@link Document}
     */
    Document deleteAsset(String assetHandle, S7Config s7Config);

    /**
     * Deletes a folder from Scene7.
     * 
     * @param folderHandle
     *            the handle to the folder to delete
     * @param s7Config
     *            the Scene7 configuration
     * @return the API's response wrapped in a {@link Document}
     */
    Document deleteFolder(String folderHandle, S7Config s7Config);

    /**
     * Gets all jobs that are currently active in the system.
     * 
     * @param jobHandle
     *            the handle to the job
     * @param originalName
     *            the original name of the job
     * @param s7Config
     *            the Scene7 configuration
     * @return the API's response wrapped in a {@link Document}
     */
    Document getActiveJobs(String jobHandle, String originalName, S7Config s7Config);

    /**
     * Gets assets from IPS.
     * 
     * @param assetHandles
     *            the asset handles
     * @param responseFields
     *            a list of fields and subfields for inclusion in the response
     * @param excludeFields
     *            a list of fields and subfields for exclusion from the response
     * @param s7Config
     *            the Scene7 configuration
     * @return the API's response wrapped in a {@link Document}
     */
    Document getAssets(String[] assetHandles, String[] responseFields, String[] excludeFields, S7Config s7Config);

    /**
     * Gets assets associated with the specified asset and details about their relationship.
     * 
     * @param assetHandle
     *            the handle to the asset
     * @param responseFields
     *            a list of fields and subfields for inclusion in the response
     * @param excludeFields
     *            a list of fields and subfields for exclusion from the response
     * @param s7Config
     *            the Scene7 configuration
     * @return the API's response wrapped in a {@link Document}
     */
    Document getAssociatedAssets(String assetHandle, String[] responseFields, String[] excludeFields, S7Config s7Config);

    /**
	 * Gets a user's memberships.
	 * 
	 * @param resourceResolver
	 *            resource resolver used to access Scene7 configuration
	 * @param userHandle
	 *            the handle to the user whose memberships you want to obtain
	 * @param email
	 *            the user's email
	 * @param password
	 *            the user's password
	 * @param region
	 *            the user's region
	 * @return the API's response wrapped in a {@link Document}
	 */
	Document getCompanyMembership(ResourceResolver resourceResolver,
			String userHandle, String email, String password, String region);

    /**
     * Gets folders and sub-folders in a hierarchical tree structure. The getFolderTree response is limited to a maximum of 100000 folders.
     * 
     * @param folderPath
     *            the root folder to retrieve folders and all subfolders to the leaflevel; if excluded, the company root is used
     * @param depth
     *            a value of zero gets the top-level folder; any other value specifies the depth to descend into the tree
     * @param responseFields
     *            a list of fields and subfields for inclusion in the response
     * @param excludeFields
     *            a list of fields and subfields for exclusion from the response
     * @param s7Config
     *            the Scene7 configuration
     * @return the API's response wrapped in a {@link Document}
     */
    Document getFolderTree(String folderPath, int depth, String[] responseFields, String[] excludeFields, S7Config s7Config);

    /**
     * Gets the details of job logs of a company.
     * 
     * @param jobHandle
     *            a handle to an active or completed job
     * @param originalName
     *            the original name of the job log
     * @param s7Config
     *            the Scene7 configuration
     * @return the API's response wrapped in a {@link Document}
     */
    Document getJobLogDetails(String jobHandle, String originalName, S7Config s7Config);

    /**
     * Gets property sets associated with a type handle.
     * 
     * @param appSettingsTypeHandle
     *            the handle to the property set type
     * @param s7Config
     *            the Scene7 configuration
     * @return the API's response wrapped in a {@link Document}
     */
    Document getPropertySets(String appSettingsTypeHandle, S7Config s7Config);

    /**
     * Gets the property set types associated with the specified company from the {@code S7Config} object.
     * 
     * @param s7Config
     *            the Scene7 configuration
     * @return the API's response wrapped in a {@link Document}
     */
    Document getPropertySetTypes(S7Config s7Config);

    /**
     * Gets the SPS image presets
     * 
     * @param s7Config
     *            the Scene7 configuration
     * @return the API's response wrapped in a {@link Document}
     */
    Document getImagePresets(S7Config s7Config);
    
    /**
	 * Gets information about a user.
	 * 
	 * @param resourceResolver
	 *            resource resolver used to access Scene7 configuration
	 * @param email
	 *            the user's email
	 * @param password
	 *            the user's password
	 * @param region
	 *            the user's region
	 * @return the API's response wrapped in a {@link Document}
	 */
	Document getUserInfo(ResourceResolver resourceResolver, String email,
			String password, String region);

    /**
     * Searches for assets based on one or more filter criteria.
     * 
     * @param folder
     *            the root path for searching for assets; if omitted, the
     *            company root folder will be used
     * @param includeSubfolders
     *            if true, the search includes sub-folders
     * @param published
     *            if true, the search will include only published assets
     * @param assetTypes
     *            array of Asset Types to include in search
     * @param assetSubTypes
     *            array of sub Asset Types to include in search
     * @param responseFields
     *            a list of fields and subfields for inclusion in the response
     * @param excludeFields
     *            a list of fields and subfields for exclusion from the response
     * @param recordsPerPage
     *            maximum number of results to return
     * @param resultsPage
     *            specifies the page of results to return, based on
     *            {@code recordsPerPage} page size
     * @param s7Config
     *            the Scene7 configuration
     * @return the API's response wrapped in a {@link Document}
     * @deprecated * @deprecated No longer supported. Replaced by
     *             {@link com.day.cq.dam.scene7.api.Scene7APIClient#searchAssets(String, Boolean, Boolean, String[], String[], String[], String[], int, int, String, S7Config)}
     */
    @Deprecated
    Document searchAssets(String folder, Boolean includeSubfolders,
            Boolean published, String[] assetTypes, String[] assetSubTypes,
            String[] responseFields, String[] excludeFields,
            int recordsPerPage, int resultsPage, S7Config s7Config);

    /**
     * Searches for assets based on one or more filter criteria.
     * 
     * @param folder
     *            the root path for searching for assets; if omitted, the
     *            company root folder will be used
     * @param includeSubfolders
     *            if true, the search includes sub-folders
     * @param published
     *            if true, the search will include only published assets
     * @param assetTypes
     *            array of Asset Types to include in search
     * @param assetSubTypes
     *            array of sub Asset Types to include in search
     * @param responseFields
     *            a list of fields and subfields for inclusion in the response
     * @param excludeFields
     *            a list of fields and subfields for exclusion from the response
     * @param recordsPerPage
     *            maximum number of results to return
     * @param resultsPage
     *            specifies the page of results to return, based on
     *            {@code recordsPerPage} page size
     * @param fileName
     *            the filename for search asset
     * @param s7Config
     *            the Scene7 configuration
     * @return the API's response wrapped in a {@link Document}
     */
    Document searchAssets(String folder, Boolean includeSubfolders, Boolean published, String[] assetTypes, String[] assetSubTypes,
            String[] responseFields, String[] excludeFields, int recordsPerPage, int resultsPage, String fileName, S7Config s7Config);

    /**
     * Searches the metadata index repository for the given search terms.
     * 
     * @param folder
     *            the root path for searching for assets; if omitted, the company root folder will be used
     * @param includeSubfolders
     *            if true, the search includes sub-folders
     * @param assetTypes
     *            array of Asset Types to include in search
     * @param assetSubTypes
     *            array of sub Asset Types to include in search
     * @param published
     *            if true, the search will include only published assets
     * @param conditions
     *            one or more query conditions that define the search criteria
     * @param s7Config
     *            the Scene7 configuration
     * @return the API's response wrapped in a {@link Document}
     */
    Document searchAssetsByMetadata(String folder, Boolean includeSubfolders, String[] assetTypes, String[] assetSubTypes,
            Boolean published, MetadataCondition[] conditions, S7Config s7Config);

    /**
     * Searches the metadata index repository for the given search terms, support loading assets by page
     * 
     * @param folder
     *            the root path for searching for assets; if omitted, the company root folder will be used
     * @param includeSubfolders
     *            if true, the search includes sub-folders
     * @param assetTypes
     *            array of Asset Types to include in search
     * @param assetSubTypes
     *            array of sub Asset Types to include in search
     * @param published
     *            if true, the search will include only published assets
     * @param conditions
     *            one or more query conditions that define the search criteria
     * @param recordsPerPage
     *            maximum number of results to return
     * @param resultsPage
     *            specifies the page of results to return, based on {@code recordsPerPage} page size
     * @param s7Config
     *            the Scene7 configuration
     * @return the API's response wrapped in a {@link Document}
     */
    Document searchAssetsByMetadata(String folder, Boolean includeSubfolders, String[] assetTypes,
            String[] assetSubTypes, Boolean published, MetadataCondition[] conditions, int recordsPerPage,
            int resultsPage, S7Config s7Config);

    /**
     * Submits a job to the system.
     * 
     * @param jobName
     *            the name of the job
     * @param jobType
     *            the job's type
     * @param s7Config
     *            the Scene7 configuration
     * @return the API's response wrapped in a {@link Document}
     * @see Scene7JobType
     */
    Document submitJob(String jobName, Scene7JobType jobType, S7Config s7Config);

    /**
     * Retrieves the system properties XML document served by the GetSystemProperties SPS servlet.
     * 
     * @param s7Config
     * @return the SPS system properties wrapped in a {@link Document}
     */
    Document getSystemProperties(S7Config s7Config);

    /**
     * Provides a way to get the originally uploaded file to the Scene7 servers after an asset has been processed.
     * 
     * @param assetHandle
     *            the asset's handle (unique id generated by Scene7)
     * @param s7Config
     *            the Scene7 configuration
     * @return an InputStream from which the file can be retrieved; the InputStream can be null if any error occurred
     */
    InputStream getOriginalFile(String assetHandle, S7Config s7Config);

    /**
	 * Provides a way to get the published image for a Scene7 asset
	 * 
	 * @deprecated Incompatible with Scene7 IPS SSL endpoint. Please use
	 *             getOriginalFile() instead.
	 * 
	 * @param asset
	 *            the <code>Scene7Asset</code> asset object
	 * @param s7Config
	 *            the <code>S7Config</code> configuration
	 * @return an <code>InputStream</code> of the published image asset
	 */
	@Deprecated
    InputStream getIPSFile(Scene7Asset asset, S7Config s7Config);
    
    /**
     * Retrieves a JPEG image in an InputStream that can be used as a thumbnail in CQ DAM. The image has the same size as the originally
     * uploaded asset from Scene7.
     * 
     * @param scene7Asset
     *            the Scene7 asset abstractization
     * @param s7Config
     *            the Scene7 configuration
     * @return an InputStream from which the file can be retrieved; the InputStream can be null if any error occurred
     */
    InputStream getThumbnailForAsset(Scene7Asset scene7Asset, S7Config s7Config);

	/**
	 * Sets the publish state of a given Scene7 asset
	 * 
	 * @param assetHandle
	 *            the Scene7 asset handle
	 * @param publishState
	 *            the asset publish state
	 * @param s7Config
	 *            the Scene7 configuration
	 * @return
	 */
	Document setAssetPublishState(String assetHandle, S7Config s7Config,
			String publishState);
}