/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.dam.scene7.api;

import java.util.ArrayList;
import java.util.List;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;

import com.day.cq.dam.scene7.api.model.Scene7Asset;
import com.day.cq.dam.scene7.api.model.Scene7Folder;
import com.day.cq.dam.scene7.api.model.Scene7PropertySet;
import com.day.cq.dam.scene7.api.model.UploadJobDetail;
import com.day.cq.dam.scene7.impl.MetadataCondition;

/**
 * The <code>Scene7Service</code> provides methods for interacting with the official Scene7 API.
 */
public interface Scene7Service {

    // Assoc types
    /**
     * Scene7 specification used in getAssociatedAssets - Array of set and template assets containing the specified asset.
     */
    public static final int CONTAINER = 1;
    /**
     * Scene7 specification used in getAssociatedAssets - Array of assets contained by the specified set or template asset.
     */
    public static final int MEMBER = 2;
    /**
     * Scene7 specification used in getAssociatedAssets - Array of assets that own the specified asset.
     */
    public static final int OWNER = 3;
    /**
     * Scene7 specification used in getAssociatedAssets - Array of assets that are derivatives of the specified asset.
     */
    public static final int DERIVED = 4;
    /**
     * Scene7 specification used in getAssociatedAssets - Array of assets that were used to generate the specified asset.
     */
    public static final int GENERATOR = 5;
    /**
     * Scene7 specification used in getAssociatedAssets - Array of assets that were generated from the specified asset.
     */
    public static final int GENERATED = 6;

    /**
     * Returns the domain name of the S7 publish server
     * 
     * @param s7Config
     * @return A string
     */
    String getPublishServer(S7Config s7Config);

    /**
     * Returns the application property handle.
     * 
     * @param s7Config
     *            the Scene7 configuration
     * @return the application property handle
     */
    String getApplicationPropertyHandle(S7Config s7Config);

    /**
     * Returns the video encoder preset type handle.
     * 
     * @param s7Config
     *            the Scene7 configuration
     * @return the video encoder preset type handle
     */
    String getVideoEncoderPresetTypeHandle(S7Config s7Config);

    /**
     * Returns the video encoder preset type handle.
     * 
     * @param typeHandle
     *            - the type handle
     * @param s7Config
     *            the Scene7 configuration
     * @return the list of property sets
     */
    List<Scene7PropertySet> getPropertySets(String typeHandle, S7Config s7Config);

    /**
	 * Returns a unique user handle from Scene7.
	 * 
	 * @param resourceResolver
	 *            resource resolver used to access Scene7 configuration
	 * @param email
	 *            E-Mail address
	 * @param password
	 *            Password
	 * @param region
	 *            Region of the Scene7 service
	 * @return A unique user handle
	 */
	String getUserHandle(ResourceResolver resourceResolver, String email,
			String password, String region);

    /**
	 * Returns a JSON representation of the company membership information
	 * provided by Scene7. The information includes a company handle, a company
	 * name and an asset root path.
	 * <p>
	 * For example:<br>
	 * <code>
	 * {<br>
	 * handle: 'jlkj98',<br>
	 * name: 'Acme Ltd.',<br>
	 * rootPath: '/MyAssets'<br>
	 * }
	 * </code>
	 * </p>
	 * 
	 * @param resourceResolver
	 *            resource resolver used to access Scene7 configuration
	 * @param userHandle
	 *            Unique user handle
	 * @param email
	 *            E-Mail address
	 * @param password
	 *            Password
	 * @param region
	 *            Region of the Scene7 service
	 * @return A JSON string
	 */
	String getCompanyMembership(ResourceResolver resourceResolver,
			String userHandle, String email, String password, String region);

    /**
     * Returns a <code>Scene7Folder</code> holding a tree structure
     * 
     * @param folderPath
     *            - the top folder of the request - does not need to be the root.
     * @param depth
     *            - depth of the returned folder tree.
     * @param responseFields
     *            - fields to be filled in for each node in the tree
     * @param excludeFields
     *            - fields to be excluded in each node of the tree
     * @param s7Config
     * @return A <code>Scene7Folder</code> object containing the requested folder structure
     */
    Scene7Folder getFolderTree(String folderPath, int depth, String[] responseFields, String[] excludeFields, S7Config s7Config);

    /**
     * Returns s7 folder handle for the associated cq folder
     * 
     * @param cqFolderPath
     *            - the path the target cq folder targeted in the s7 folder organization
     * @param s7Config
     * @return folder handle or null if the folder does not exist in s7.
     */
    String getFolderHandle(String cqFolderPath, S7Config s7Config);

    /**
     * Returns a String success/ failure,, note that the underlying S7 API call does not provide a data response.
     * 
     * @param folderHandle
     *            - folder to be deleted
     * @param s7Config
     *            - s7 config based on cloud service config info.
     * @return String success / failure
     */
    String deleteFolder(String folderHandle, S7Config s7Config);

    /**
     * Returns a String success/ failure,, note that the underlying S7 API call does not provide a data response.
     * 
     * @param assetHandle
     *            - asset to be deleted
     * @param s7Config
     *            - s7 config based on cloud service config info.
     * @return String success / failure
     */
    String deleteAsset(String assetHandle, S7Config s7Config);

    /**
     * Returns a DOM list with search results (from metadata)
     * 
     * @param folder
     *            - all matched assets will be in this folder
     * @param includeSubfolders
     *            - whether to recursively search for assets
     * @param assetTypes
     *            - all matched assets will be one of these types
     * @param assetSubTypes
     *            array of sub Asset Types to include in search
     * @param published
     *            - whether the matched assets must be marked for publish (only marked - not necessarily actually published)
     * @param conditions
     *            - metadata conditions for search
     * @param s7Config
     * @return A list of assets; the list can be empty if no assets have been found.
     */
    List<Scene7Asset> searchAssetsByMetadata(String folder, Boolean includeSubfolders, String[] assetTypes, String[] assetSubTypes,
            Boolean published, MetadataCondition[] conditions, S7Config s7Config);

    /**
     * Returns a DOM list with search results (from metadata), support specify 
     * 
     * @param folder
     *            - all matched assets will be in this folder
     * @param includeSubfolders
     *            - whether to recursively search for assets
     * @param assetTypes
     *            - all matched assets will be one of these types
     * @param assetSubTypes
     *            array of sub Asset Types to include in search
     * @param published
     *            - whether the matched assets must be marked for publish (only marked - not necessarily actually published)
     * @param conditions
     *            - metadata conditions for search
     * @param recordsPerPage
     *            maximum number of results to return
     * @param resultsPage
     *            specifies the page of results to return, based on {@code recordsPerPage} page size
     * @param s7Config
     * @return A list of assets; the list can be empty if no assets have been found.
     */
    List<Scene7Asset> searchAssetsByMetadata(String folder, Boolean includeSubfolders, String[] assetTypes,
            String[] assetSubTypes, Boolean published, MetadataCondition[] conditions, int recordsPerPage,
            int resultsPage, S7Config s7Config);

    /**
     * Returns a <code>Scene7Asset</code> list, based on a list of asset handles
     * 
     * @param assetHandles
     *            - the assets to retrieve
     * @param responseFields
     *            - fields to be filled in for each node in the tree
     * @param excludeFields
     *            - fields to be excluded in each node of the tree
     * @param s7Config
     * @return A <code>Scene7Asset</code> list
     */
    List<Scene7Asset> getAssets(String[] assetHandles, String[] responseFields, String[] excludeFields, S7Config s7Config);

    /**
     * Returns an Arraylist of job details objects
     * 
     * @param jobHandle
     *            - the handle for the job to get details.
     * @param s7Config
     * @return An AssetList of UploadJobDetail
     */
    ArrayList<UploadJobDetail> getMultiFileJobLogDetails(String jobHandle, S7Config s7Config);

    /**
     * @param jobHandle
     *            of the job
     * @param s7Config
     * @return <code>true</code> if active job with this jobHandle or orginalName is active, <code>false</code> otherwise
     */
    boolean isJobActiveByJobHandle(String jobHandle, S7Config s7Config);

    /**
     * @param originalName
     *            of the job
     * @param s7Config
     * @return <code>true</code> if active job with this jobHandle or orginalName is active, <code>false</code> otherwise
     */
    boolean isJobActiveByOriginalName(String originalName, S7Config s7Config);

    /**
     * Returns a list of Scene7 asset handles(Strings) for the given job handle
     * 
     * @param jobHandle
     *            - the handle for the job to get details.
     * @param s7Config
     * @return A list containing asset handles strings
     */
    List<String> getJobLogDetails(String jobHandle, S7Config s7Config);

    /**
     * Returns a list of Scene7 image preset names for the given {@code S7Config}
     * 
     * @param s7Config
     * @return
     *      a list containing the names of the image presets defined in SPS
     */
    List<String> getImagePresets(S7Config s7Config);
    
    /**
     * Retrieves the server used for serving and manipulating flash templates, depending on the Scene7 configuration.
     * 
     * @param s7Config
     *            the Scene7 configuration
     * @return a String containing the server's URL
     */
    String getFlashTemplatesServer(S7Config s7Config);

    /**
     * Retrieves a list of {@code Scene7Asset}s from the Scene7 server.
     * 
     * @param folder
     *            the folder from which the assets are retrieved
     * @param includeSubfolders
     *            whether to recursively search for assets in sub-folders of the folder where the search is performed
     * @param published
     *            whether the matched assets must be marked for publish (only marked - not necessarily actually published)
     * @param assetTypes
     *            all matched assets will be one of these types
     * @param assetSubTypes
     *            array of sub Asset Types to include in search
     * @param responseFields
     *            the response fields that should be used to populate the {@code Scene7Asset}'s attributes
     * @param excludeFields
     *            fields to be excluded from the response
     * @param s7Config
     *            the Scene7 configuration
     * @return a List of assets; can be empty if no assets have been found
     */
    List<Scene7Asset> searchAssets(String folder, Boolean includeSubfolders, Boolean published, String[] assetTypes,
            String[] assetSubTypes, String[] responseFields, String[] excludeFields, S7Config s7Config);

    /**
     * Retrieves a list of {@code Scene7Asset}s from the Scene7 server.
     * 
     * @param folder
     *            the folder from which the assets are retrieved
     * @param includeSubfolders
     *            whether to recursively search for assets in sub-folders of the folder where the search is performed
     * @param published
     *            whether the matched assets must be marked for publish (only marked - not necessarily actually published)
     * @param assetTypes
     *            all matched assets will be one of these types
     * @param assetSubTypes
     *            array of sub Asset Types to include in search
     * @param responseFields
     *            the response fields that should be used to populate the {@code Scene7Asset}'s attributes
     * @param excludeFields
     *            fields to be excluded from the response
     * @param recordsPerPage
     *            maximum number of results to return
     * @param resultsPage
     *            specifies the page of results to return, based on {@code recordsPerPage} page size
     * @param s7Config
     *            the Scene7 configuration
     * @return a List of assets; can be empty if no assets have been found
     */
    List<Scene7Asset> searchAssets(String folder, Boolean includeSubfolders, Boolean published, String[] assetTypes,
            String[] assetSubTypes, String[] responseFields, String[] excludeFields, final int recordsPerPage,
            final int resultsPage, S7Config s7Config);
    
    /**
     * Retrieve a list of {@code Scene7Asset}s from the Scene7 server by
     * filename.
     * 
     * @param folder
     * @param includeSubfolders
     * @param published
     * @param filename
     * @param s7Config
     * @return
     */
    List<Scene7Asset> searchAssetsByFilename(String folder,
            Boolean includeSubfolders, Boolean published, String filename,
            S7Config s7Config);

    /**
     * Retrieves the sub-assets and the origiator asset associated with the
     * given asset
     * 
     * @param asset
     *            - the asset to get the sub-assets and generator from
     * @param s7Config
     *            - the Scene7 configuration
     * @return the original <code>Scene7Asset</code> object updated with the
     *         sub-assets and the originator asset
     */
    Scene7Asset getAssociatedAssets(Scene7Asset asset, S7Config s7Config);

	/**
	 * Retrieves the originator asset associated with the given asset
	 * 
	 * @param asset
	 *            - the asset to get the originator asset
	 * @param s7Config
	 *            - the Scene7 configuration
	 * @return the original <code>Scene7Asset</code> object updated with the
	 *         originator asset
	 */
	Scene7Asset getMasterAsset(Scene7Asset asset, S7Config s7Config);

    /**
     * Retrieves the server handling FXG rendering.
     * 
     * @param s7Config
     *            the Scene7 configuration
     * @return a String containing the FXG server URL
     */
    String getFxgServer(S7Config s7Config);

	/**
	 * Set the publish state of a given Scene7 asset
	 * 
	 * @param resource
	 *            activated resource
	 * @param markForPublish
	 *            value of markForPublish flag
	 * @param s7Config
	 *            the Scene7 configuration
	 * @return String success / failure
	 */
	String setAssetPublishState(Resource resource, boolean markForPublish,
			S7Config s7Config);

	/**
	 * Set the publish state of a given Scene7 asset
	 * 
	 * @param assetHandle
	 *            unique asset identifier
	 * @param markForPublish
	 *            value of markForPublish flag
	 * @param s7Config
	 *            the Scene7 configuration
	 * @return String success / failure
	 */
	String setAssetPublishState(String assetHandle, boolean markForPublish,
			S7Config s7Config);
}
