/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.dam.scene7.api.model;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.day.cq.dam.scene7.api.constants.Scene7AssetType;
import com.day.cq.dam.scene7.impl.utils.Scene7AssetUtils;

/**
 * Default implementation for a Scene7Asset
 */
public class Scene7AssetImpl implements Scene7Asset {

    private static final long serialVersionUID = 5536087022446822682L;

    /**
     * Scene7 asset property keys
     */
    public enum Scene7AssetProperty {
        ASSET_TYPE_STRING,
        SCENE7_ASSET_TYPE,
        ASSET_HANDLE,
        NAME,
        FILE_NAME,
        FOLDER,
        FOLDER_HANDLE,
        CREATED_BY,
        MODIFIED_BY,
        VIDEO_ENCODING_PRESET_ID,
        CREATED_DATE,
        MODIFIED_DATE,
        ORIGINAL_FILE,
        ORIGINAL_PATH,
        WIDTH,
        HEIGHT,
        URL_MODIFIER,
        READY_TO_PUBLISH,
        FILE_SIZE,
        VIEWER_PRESET_TYPE,
        VIEWER_PRESET_CONFIGURATION_SETTINGS,
        SUB_ASSETS,
        ORIGINATOR,
        ASSET_PROPERTIES
    }

    /**
     * <code>Map</code> containing the asset's main attributes
     */
    private Map<Object, Object> assetAttributes;

    /**
     * Creates a new <code>Scene7Asset</code> object
     * 
     * @param assetAttributes
     *            a <code>Map</code> object containing the new Scene7 asset's main attributes
     */
    public Scene7AssetImpl(Map<Object, Object> assetAttributes) {

        if (assetAttributes != null) {
            // use the received map only if it is non-null
            this.assetAttributes = assetAttributes;
        } else {
            // create a new map
            this.assetAttributes = new HashMap<Object, Object>();
        }

        // initialize the sub-asset list
        this.assetAttributes.put(Scene7AssetProperty.SUB_ASSETS, new ArrayList<Scene7Asset>());
    }

    /**
     * Copy constructor useful for changing the properties for an already parsed <code>Scene7Asset</code>
     * 
     * @param scene7Asset
     *            the original <code>Scene7Asset</code>
     * @param overrideProperties
     *            a <code>Map</code> containing the properties that need to be updated
     */
    public Scene7AssetImpl(Scene7Asset scene7Asset, Map<Object, Object> overrideProperties) {
        // copy the asset's properties into a map
        Map<Object, Object> originalPropertyMap = getPropertiesMap(scene7Asset);

        // override the properties in the map
        if (overrideProperties != null) {
            originalPropertyMap.putAll(overrideProperties);
        }

        // use the resulting property map for this asset
        this.assetAttributes = originalPropertyMap;
    }

    public Scene7AssetType getAssetType() {
        return getProperty(assetAttributes, Scene7AssetProperty.SCENE7_ASSET_TYPE, Scene7AssetType.class);
    }

    public String getAssetTypeStr() {
        return getProperty(assetAttributes, Scene7AssetProperty.ASSET_TYPE_STRING, String.class);
    }

    public String getName() {
        return getProperty(assetAttributes, Scene7AssetProperty.NAME, String.class);
    }

    public String getAssetHandle() {
        return getProperty(assetAttributes, Scene7AssetProperty.ASSET_HANDLE, String.class);
    }

    public String getFileName() {
        return getProperty(assetAttributes, Scene7AssetProperty.FILE_NAME, String.class);
    }

    public String getFolder() {
        return getProperty(assetAttributes, Scene7AssetProperty.FOLDER, String.class);
    }

    public String getFolderHandle() {
        return getProperty(assetAttributes, Scene7AssetProperty.FOLDER_HANDLE, String.class);
    }

    public String getCreatedBy() {
        return getProperty(assetAttributes, Scene7AssetProperty.CREATED_BY, String.class);
    }

    public String getModifiedBy() {
        return getProperty(assetAttributes, Scene7AssetProperty.MODIFIED_BY, String.class);
    }

    public Date getCreatedDate() {
        return getProperty(assetAttributes, Scene7AssetProperty.CREATED_DATE, Date.class);
    }

    public Date getModifiedDate() {
        return getProperty(assetAttributes, Scene7AssetProperty.MODIFIED_DATE, Date.class);
    }

    public String getOriginalFile() {
        return getAssetProperties().get("originalFile");
    }

    public String getOriginalPath() {
        return getAssetProperties().get("originalPath");
    }

    /**
     * Returns an unmodifiable view of the internal sub-assets list
     */
    @SuppressWarnings("unchecked")
    public List<Scene7Asset> getSubAssets() {
        List<Scene7Asset> subAssets = getProperty(assetAttributes, Scene7AssetProperty.SUB_ASSETS, List.class);
        if (subAssets != null) {
            subAssets = Collections.unmodifiableList(subAssets);
        }

        return subAssets;
    }

    public Scene7Asset getOriginatorAsset() {
        return getProperty(assetAttributes, Scene7AssetProperty.ORIGINATOR, Scene7Asset.class);
    }

    public void updateOriginatorAsset(Scene7Asset originator) {
        this.assetAttributes.put(Scene7AssetProperty.ORIGINATOR, originator);
    }

    @SuppressWarnings("unchecked")
    public void addSubAsset(Scene7Asset subAsset) {
        List<Scene7Asset> subAssets = getProperty(assetAttributes, Scene7AssetProperty.SUB_ASSETS, List.class);
        if (subAssets != null) {
            subAssets.add(subAsset);
        }
    }

    public Long getWidth() {
        return (Long) assetAttributes.get(Scene7AssetProperty.WIDTH);
    }

    public Long getHeight() {
        return (Long) assetAttributes.get(Scene7AssetProperty.HEIGHT);
    }

    public String getUrlModifier() {
        return getAssetProperties().get("urlModifier");
    }

    @SuppressWarnings("unchecked")
    public Map<String, String> getViewerPresetConfigurationSettings() {
        return getProperty(assetAttributes, Scene7AssetProperty.VIEWER_PRESET_CONFIGURATION_SETTINGS, Map.class);
    }

    public String getViewerPresetType() {
        return getAssetProperties().get("type");
    }

    public String getRootFolder() {
        String folder = getProperty(assetAttributes, Scene7AssetProperty.FOLDER, String.class);
        if (folder != null) {
            int idx = folder.indexOf("/");
            if (idx > -1) {
                String root = folder.substring(0, idx + 1);
                if (!root.endsWith("/")) {
                    root += "/";
                }
                return root;
            }
        }
        return "";
    }

    public boolean isPublished() {
        return Boolean.parseBoolean(getProperty(assetAttributes, Scene7AssetProperty.READY_TO_PUBLISH, String.class));
    }

    public Integer getFileSize() {
        return Scene7AssetUtils.parseInt(getAssetProperties().get("fileSize"));
    }

    public String getVideoEncodingPresetId() {
        return getProperty(assetAttributes, Scene7AssetProperty.VIDEO_ENCODING_PRESET_ID, String.class);
    }

    /**
     * Returns one of the properties stored in a property map, while also converting to a specified type if possible
     * 
     * @param map
     *            the map from which to extract the property value
     * @param key
     *            the key under which the data is stored
     * @param convertTo
     *            the type to which conversion will be attempted
     * 
     * @return a property converted to the requested type if successful, null otherwise
     */
    private <T> T getProperty(Map<?, ?> map, Object key, Class<T> convertTo) {
        Object propValue = map.get(key);
        T convertedPropValue = null;
        if (propValue != null && convertTo.isAssignableFrom(propValue.getClass())) {
            convertedPropValue = convertTo.cast(propValue);
        }
        return convertedPropValue;
    }

    /**
     * Creates a <code>Map</code> that contains the properties of the received <code>Scene7Asset</code>
     * 
     * @param scene7Asset
     *            a <code>Scene7Asset</code> implementation
     * 
     * @return a <code>Map</code> containing the properties of the Scene7 asset
     */
    private Map<Object, Object> getPropertiesMap(Scene7Asset scene7Asset) {
        Map<Object, Object> propertiesMap = new HashMap<Object, Object>();

        // popuplate a map based on the asset's public interface
        propertiesMap.put(Scene7AssetProperty.ASSET_TYPE_STRING, scene7Asset.getAssetTypeStr());
        propertiesMap.put(Scene7AssetProperty.SCENE7_ASSET_TYPE, scene7Asset.getAssetType());
        propertiesMap.put(Scene7AssetProperty.ASSET_HANDLE, scene7Asset.getAssetHandle());
        propertiesMap.put(Scene7AssetProperty.NAME, scene7Asset.getName());
        propertiesMap.put(Scene7AssetProperty.FILE_NAME, scene7Asset.getFileName());
        propertiesMap.put(Scene7AssetProperty.FOLDER, scene7Asset.getFolder());
        propertiesMap.put(Scene7AssetProperty.FOLDER_HANDLE, scene7Asset.getFolderHandle());
        propertiesMap.put(Scene7AssetProperty.CREATED_BY, scene7Asset.getCreatedBy());
        propertiesMap.put(Scene7AssetProperty.MODIFIED_BY, scene7Asset.getModifiedBy());
        propertiesMap.put(Scene7AssetProperty.VIDEO_ENCODING_PRESET_ID, scene7Asset.getVideoEncodingPresetId());
        propertiesMap.put(Scene7AssetProperty.CREATED_DATE, scene7Asset.getCreatedDate());
        propertiesMap.put(Scene7AssetProperty.MODIFIED_DATE, scene7Asset.getModifiedDate());
        propertiesMap.put(Scene7AssetProperty.ORIGINAL_FILE, scene7Asset.getOriginalFile());
        propertiesMap.put(Scene7AssetProperty.ORIGINAL_PATH, scene7Asset.getOriginalPath());
        propertiesMap.put(Scene7AssetProperty.WIDTH, scene7Asset.getWidth());
        propertiesMap.put(Scene7AssetProperty.HEIGHT, scene7Asset.getHeight());
        propertiesMap.put(Scene7AssetProperty.URL_MODIFIER, scene7Asset.getUrlModifier());
        propertiesMap.put(Scene7AssetProperty.READY_TO_PUBLISH, scene7Asset.isPublished());
        propertiesMap.put(Scene7AssetProperty.FILE_SIZE, scene7Asset.getFileSize());
        propertiesMap.put(Scene7AssetProperty.VIEWER_PRESET_TYPE, scene7Asset.getViewerPresetType());
        propertiesMap.put(Scene7AssetProperty.VIEWER_PRESET_CONFIGURATION_SETTINGS, scene7Asset.getViewerPresetConfigurationSettings());
        propertiesMap.put(Scene7AssetProperty.SUB_ASSETS, scene7Asset.getSubAssets());
        propertiesMap.put(Scene7AssetProperty.ORIGINATOR, scene7Asset.getOriginatorAsset());
        propertiesMap.put(Scene7AssetProperty.ASSET_PROPERTIES, scene7Asset.getAssetProperties());
        return propertiesMap;
    }

    @SuppressWarnings("unchecked")
    public Map<String, String> getAssetProperties() {
        return (Map<String, String>) assetAttributes.get(Scene7AssetProperty.ASSET_PROPERTIES);
    }
}
