/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.mcm.landingpage.parser.taghandlers.mbox;

import com.day.cq.dam.indd.PageComponent;
import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.designimporter.DesignImportException;
import com.day.cq.wcm.designimporter.UnsupportedTagContentException;
import com.day.cq.wcm.designimporter.api.CanvasBuilder;
import com.day.cq.wcm.designimporter.api.ContainerComponentProvider;
import com.day.cq.wcm.designimporter.parser.HTMLContentType;
import com.day.cq.wcm.designimporter.parser.taghandlers.AbstractTagHandler;
import com.day.cq.wcm.designimporter.parser.taghandlers.CanvasComponentBuilder;
import com.day.cq.wcm.designimporter.parser.taghandlers.DefaultTagHandler;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.event.Event;
import org.osgi.service.event.EventConstants;
import org.osgi.service.event.EventHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.Attributes;

import javax.jcr.RepositoryException;
import java.util.ArrayList;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.List;

/**
 * This tag handler handles the experience tags encountered within the target component tag.
 *
 * <p>
 * This tag handler is registered against the markup of type < section data-cq-component="offer" >
 * in {@link com.day.cq.wcm.designimporter.impl.TagHandlerProviderImpl}. That implies
 * this tag handler will come into action whenever the design importer framework
 * encounters an HTML tag matching the rule this handler is defined against.
 * </p>
 */
public class MBoxExperienceTagHandler extends AbstractTagHandler implements EventHandler,ContainerComponentProvider {

    private String segment;

    private ServiceRegistration serviceRegistration;

    private Logger logger = LoggerFactory.getLogger(MBoxExperienceTagHandler.class);

    /**
     * @return true, if this tag handler is handling the default offer. The default offer is the one
     * the component tree for which is persisted under the target component. The content defined by
     * the default offer is displayed in the default case.
     */
    public boolean isDefaultExperience() {
        return isDefaultExperience;
    }

    private boolean isDefaultExperience;

    private boolean componentGenerationRequired;
    
    private String resourceType;

    private List<PageComponent> offerComponents;

    @Override
    public void beginHandling(String uri, String localName, String qName, Attributes atts) throws DesignImportException {
        super.beginHandling(uri, localName, qName, atts);

        String cqExperience = atts.getValue("data-cq-experience");
        if ("default".equalsIgnoreCase(cqExperience)) {
            isDefaultExperience = true;
        }

        this.segment = atts.getValue("data-cq-segment");

    }

    @Override
    public void startElement(String uri, String localName, String qName, Attributes atts) throws DesignImportException,
            UnsupportedTagContentException {
        super.startElement(uri, localName, qName, atts);

        if (delegate instanceof DefaultTagHandler)
            componentGenerationRequired = true;
    }

    @Override
    public void characters(char[] ch, int start, int length) throws DesignImportException {
        super.characters(ch, start, length);
        if (delegate == null && new String(ch, start, length).trim().length() > 0) // Only WS characters allowed. Any text is not tolerable.
            componentGenerationRequired = true;
    }

    @Override
    public void endHandling(String uri, String localName, String qName) throws DesignImportException {
        super.endHandling(uri, localName, qName);

        if(componentGenerationRequired) {
            // Wrap pageComponents into a top level component which has the "generated" resourceType.
            resourceType = deriveResourceType();
            PageComponent wrapper = pageBuilder.createComponent(resourceType);
            wrapper.getChildComponents().addAll(getPageComponents());
            pageComponents = new ArrayList<PageComponent>();
            pageComponents.add(wrapper);

            // Register for callback
            String[] topics = new String[] {CanvasBuilder.EVENT_COMPLETE};
            Dictionary props = new Hashtable();
            props.put(EventConstants.EVENT_TOPIC, topics);
            if(designImporterContext.bundleContext != null) {
                serviceRegistration = designImporterContext.bundleContext.registerService(EventHandler.class.getName(), this , props);
            }
        }

        if (!isDefaultExperience) {
            offerComponents = new ArrayList<PageComponent>();
            offerComponents.addAll(getPageComponents());

            // Reset the page component list so that they don't get mixed with actual page components.
            pageComponents = new ArrayList<PageComponent>();
        }
    }

    @Override
    public boolean supportsContent(HTMLContentType htmlContentType) {
        return false;
    }

    /**
     * @return The list of components that should be persisted in form of campaign offers
     */
    public List<PageComponent> getOfferComponents() {
        return offerComponents;
    }

    /**
     * @return The segment associated with the offer being handled by this tag handler
     */
    public String getSegment() {
        return segment;
    }

    public void handleEvent(Event event) {
        if(componentGenerationRequired) {
            try {
                CanvasComponentBuilder mboxOfferComponentBuilder = new CanvasComponentBuilder();
                mboxOfferComponentBuilder.setResourceType(resourceType);
                mboxOfferComponentBuilder.setSession(designImporterContext.designNode.getSession());
                mboxOfferComponentBuilder.appendScriptContent(htmlBuffer.toString());
                mboxOfferComponentBuilder.build();
            } catch (RepositoryException e) {
                logger.error("Error occurred generating the anybox offer component");
            }
        }

        // In the end, un-register
        if(serviceRegistration != null) {
            serviceRegistration.unregister();
        }
    }
    
    private String deriveResourceType() {
        Page page = designImporterContext.page;
        String htmlName = designImporterContext.htmlName;
        String uniqueId = page.getPath().replace("/content/campaigns/", "").replace('/', '-') + "-" + htmlName;

        String mboxResourceType = null;
        try {
            String pageName = page.getPath().replaceAll(".*/",""); // Strip off chars before the last slash
            String strippedPath = page.getPath().replace("/content/campaigns/", "");
            int indexOfSlash = strippedPath.indexOf('/');
            String brand = strippedPath.replaceFirst("/.*",""); // Strip off chars after the first slash
            mboxResourceType = brand + "/components/mbox/" + uniqueId + "-" + "mbox" + designImporterContext.componentSuffixGenerator.getSuffix("mbox");
        } catch (Exception e) { // TODO: Shouldn't be Exception, if this is for NPEs, they should be handled properly.
            logger.error("Error occured while trying to deduce the mbox resource type", e);
        }
        return mboxResourceType;

    }
}
