/*************************************************************************
 * ADOBE CONFIDENTIAL
 * __________________
 * Copyright 2012 Adobe Systems Incorporated
 * All Rights Reserved.
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.personalization;

import javax.jcr.RepositoryException;
import java.util.List;

import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.commons.json.JSONObject;

import aQute.bnd.annotation.ProviderType;
import com.day.cq.wcm.api.Page;

/**
 * A service which provides access to campaigns, experiences and targeted content.
 */
@ProviderType
public interface TargetedContentManager {

    /**
     * Returns a JSON array of active teasers for a particular campaignPath (legacy teasers)
     * or location.
     *
     * The JSON for each teaser includes:
     * <ul>
     *     <li>path</li>
     *     <li>name</li>
     *     <li>title</li>
     *     <li>campainName (misspelled for legacy reasons)</li>
     *     <li>thumbnail</li>
     *     <li>id</li>
     *     <li>segments[]</li>
     *     <li>tags[]
     *         <ul>
     *             <li>name</li>
     *             <li>title</li>
     *             <li>titlePath</li>
     *             <li>path</li>
     *             <li>tagID</li>
     *         </ul>
     *     </li>
     * </ul>
     * Teasers are ordered first by campaign priority and then by experience page order.
     * This allows the "first" strategy to be used for priority selection.
     * While it might be nice to support more sophisticated strategies (for instance, a
     * more specific segment in a lower-priority campaign ahead of a more generic segment
     * in a higher-priority campaign), there is currently no support for this.
     *
     * @param resolver     A resourceResolver from the request (for access control).
     * @param campaignPath A campaign path for a legacy teaser (or null).
     * @param location     A named destination (or null for a legacy teaser).
     * @return JSONObject as described
     * @deprecated use {@link TargetedContentManager#getTeasers(ResourceResolver, String, String)} instead
     */
    JSONObject getTeaserInfo(ResourceResolver resolver, String campaignPath, String location);

    /**
     * Returns a JSON array of active teasers for a particular campaignPath (legacy teasers)
     * or location.
     *
     * @param request A sling servlet request. Cannot be <code>null</code>
     * @param campaignPath A campaign path for a legacy teaser (or null).
     * @param location A named destination (or null for a legacy teaser).
     * @return JSONObject as described in {@link #getTeaserInfo(ResourceResolver, String, String)}.
     * {@link TargetedContentManager#getTeaserInfo(ResourceResolver, String, String)}
     * @deprecated use {@link TargetedContentManager#getTeasers(SlingHttpServletRequest, String)} instead
     */
    JSONObject getTeaserInfo(SlingHttpServletRequest request, String campaignPath, String location);

    /**
     * Returns a list of teasers that are available for a location
     *
     * @param resolver a {@link ResourceResolver} used to access the repository. Cannot be <code>null</code>
     * @param location the name of the location for which to return the teasers. Cannot be <code>null</code>
     * @param pageUrl  the url of the page for which to retrieve the teasers. This url includes the extensions and the selectors.
     *                 If <code>null</code> or empty string then the teasers url won't contain the selectors
     *                 and the default teaser won't be in the list of teasers returned by this method
     * @return a list of {@link Teaser} objects or an empty list if no teasers are found
     */
    List<Teaser> getTeasers(ResourceResolver resolver, String location, String pageUrl);

    /**
     * Returns a list of teasers that are available for a location
     *
     * @param request a {@link SlingHttpServletRequest}. Cannot be <code>null</code>
     * @param location the name of the location for which to return the teasers. Cannot be <code>null</code>
     * @return a list of {@link Teaser} objects or an empty list if no teasers are found
     * @see #getTeasers(ResourceResolver, String, String)
     */
    List<Teaser> getTeasers(SlingHttpServletRequest request, String location);

    /**
     * Returns a list of campaigns sorted by priority.
     *
     * @param resourceResolver A resourceResolver from the request (for access control).
     * @return returns a list of campaigns sorted by priority.
     * @throws RepositoryException if an error occurs during repository operations.
     */
    List<Page> getCampaigns(ResourceResolver resourceResolver) throws RepositoryException;

    /**
     * Returns a list of campaigns which hold offers with the specified
     * {@code location} value.
     *
     * @param resolver ResourceResolver bound to the users session
     * @param location A named destination
     * @return A list of campaigns holding activities offers with the specified
     *         {@code location}
     * @throws RepositoryException if an error occurs during repository operations.
     */
    List<Page> getCampaigns(ResourceResolver resolver, String location) throws RepositoryException;


    /**
     * Finds the actual content paths for a given list of location names.
     *
     * @param resolver a {@link ResourceResolver} used to access the repository
     * @param locationName one or more location names for which to retrieve the path
     * @return a {@link List} of {@link Location} objects an empty list if no paths were found.
     */
    List<Location> findLocations(ResourceResolver resolver, String... locationName);

    /**
     * Retrieves the list of locations for this campaign.
     * @param resolver a {@link ResourceResolver} used to access the repo
     * @param campaignPath the path to the campaign
     * @return a {@link List} of {@link Location} objects or an empty list if the campaign doesn't have any locations (i.e
     * . offers)
     */
    List<Location> getCampaignLocations(ResourceResolver resolver, String campaignPath);
}
