/*
 * Copyright 1997-2009 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.polling.importer;

import java.io.IOException;
import java.io.InputStream;

import org.apache.commons.httpclient.Credentials;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpConnectionManager;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpMethodBase;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Properties;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Service;
import org.apache.sling.api.resource.Resource;
import org.osgi.service.component.ComponentContext;

/**
 * The <code>HttpImporter</code> is an abstract implementation of the
 * {@link Importer} service interface supporting access to data using the HTTP
 * protocol.
 * <p>
 * Extensions of this base class must implement the
 * {@link #importData(String, InputStream, String, Resource)} method to actuall
 * import the retrieved data.
 * <p>
 * Instanes of this base class <b>must</b> be initialized before being used and
 * be destroyed when not used any more. If the extesion is registered as an
 * Declarative Services Component, this initialization and destroyal is being
 * taken care of. Otherwise, the extension class must explicitly call the
 * {@link #init()} and {@link #destroy()} methods respectively.
 * <p>
 * The {@link #activate(ComponentContext)} and
 * {@link #deactivate(ComponentContext)} methods may be overwritten by extension
 * classes but the base class implementation <b>must</b> called in this case to
 * ensure proper initialization and destroyal.
 * 
 * @deprecated Please use {@link HCImporter} instead.
 */
@Component(metatype = false, componentAbstract = true)
@Service(Importer.class)
@Properties({
        @Property(name = "service.description", value = "Abstract HTTP based Importer")
})
@Deprecated
public abstract class HttpImporter implements Importer {

    /**
     * The HTTP Client used by this importer instance. This field is initialized
     * on demand by the {@link #getHttpClient()} method and cleared by the
     * {@link #destroy()} method.
     */
    private HttpClient httpClient;

    /**
     * Actually imports the data recevied from the data source into the target
     * <code>Resource</code>.
     * <p>
     * Implementations of this method need not (and should not) close the
     * <code>data</code> input stream. This will be taken care of the caller of
     * this method.
     *
     * @param scheme The scheme describing the data to import.
     * @param data The data to be imported.
     * @param characterEncoding If the data contains character data, such as
     *            XML, this parameter (may) provide the character encoding of
     *            the character data. The value of this parameter is extracted
     *            from the <code>Content-Type</code> header of the HTTP response
     *            and may thus be empty or <code>null</code> if not set by the
     *            server.
     * @param target The <code>Resource</code> into which the data is to be
     *            imported.
     * @throws IOException May be thrown if an error occurrs reading or writing
     *             the imported data.
     * @throws ImportException May be thrown in any other error case.
     */
    protected abstract void importData(String scheme, InputStream data,
            String characterEncoding, Resource target) throws IOException;

    // ---------- Importer interface

    /**
     * Imports data from the given <code>source</code> which is expected to be
     * an HTTP URL. This method uses the Apache Commons Http Client library to
     * request the source with a method provided by {@link #newHttpClient()} the
     * response stream is then provided to the
     * {@link #importData(String, InputStream, String, Resource)} method, which
     * must be implemented by the extension.
     *
     * @param scheme The scheme of the data source. This parameter is handed
     *            over to the
     *            {@link #importData(String, InputStream, String, Resource)}
     *            method.
     * @param dataSource The HTTP URL to the data to be imported.
     * @param target The <code>Resource</code> into which the data is to be
     *            imported.
     * @throws ImportException If an error occurrs while accessing the source or
     *             if this class has not beenn initialized or if an error occurs
     *             while importing the actual data through
     *             {@link #importData(String, InputStream, String, Resource)}.
     */
    public void importData(String scheme, String dataSource, Resource target) {
    	importData(scheme, dataSource, target, null, null);
    }
    
    /**
     * Imports data from the given <code>source</code> which is expected to be
     * an HTTP URL. This method uses the Apache Commons Http Client library to
     * request the source with a method provided by {@link #newHttpClient()} the
     * response stream is then provided to the
     * {@link #importData(String, InputStream, String, Resource)} method, which
     * must be implemented by the extension.
     *
     * @param scheme The scheme of the data source. This parameter is handed
     *            over to the
     *            {@link #importData(String, InputStream, String, Resource)}
     *            method.
     * @param dataSource The HTTP URL to the data to be imported.
     * @param target The <code>Resource</code> into which the data is to be
     *            imported.
     * @param login The login to authenticate the request.
     * @param password The password to authenticate the request.
     * @throws ImportException If an error occurrs while accessing the source or
     *             if this class has not beenn initialized or if an error occurs
     *             while importing the actual data through
     *             {@link #importData(String, InputStream, String, Resource)}.
     */
    public void importData(String scheme, String dataSource, Resource target, String login, String password) {

        // the method to execute
        HttpMethodBase method = newHttpMethod(dataSource);

        InputStream ins = null;
        try {

        	// if login information is present, setup the authentication on the httpclient
        	if (login != null && login.length() > 0 && password != null && password.length() > 0) {
            	getHttpClient().getParams().setAuthenticationPreemptive(true);
            	Credentials defaultcreds = new UsernamePasswordCredentials(login, password);
            	getHttpClient().getState().setCredentials(AuthScope.ANY, defaultcreds);        	
        	}

            // execute the request and check the status
            int statusCode = getHttpClient().executeMethod(method);
            switch (statusCode) {
                case HttpStatus.SC_OK:
                    // this what we expect
                    break;

                case HttpStatus.SC_NOT_MODIFIED:
                    // no change in data, so just get back out of here
                    return;

                default:
                    // this is what we don't expect
                    throw new ImportException("Failed retreiving data"
                        + method.getStatusLine());
            }

            // access the raw response stream and import it
            ins = method.getResponseBodyAsStream();
            importData(scheme, ins, method.getResponseCharSet(), target);

        } catch (HttpException e) {

            throw new ImportException("Fatal protocol violation", e);

        } catch (IOException e) {

            throw new ImportException("Fatal transport error", e);

        } catch (IllegalStateException ise) {

            throw new ImportException(
                "HttpImporter has not been initialized yet", ise);

        } catch (Exception e) {

            throw new ImportException(
                "Unexpected problem while importing data", e);

        } finally {

            // close the response input stream
            if (ins != null) {
                try {
                    ins.close();
                } catch (IOException ioe) {
                }
            }

            // Release the connection.
            method.releaseConnection();
        }
    }

    /**
     * Returns a <code>HttpClient</code> instance used by the
     * {@link #importData(String, InputStream, String, Resource)} method to
     * access the data source URL. The <code>HttpClient</code> instance is
     * backed by a <code>MultiThreadedHttpConnectionManager</code>.
     * <p>
     * This method cannot be overwritten. To configure the HttpClient, the
     * {@link #init()} method should be called with appropriate parameters which
     * are used to setup the client with an instance of the
     * <code>HttpClientParams</code> class.
     *
     * @return the <code>HttpClient</code> instance.
     * @see #init()
     */
    protected final HttpClient getHttpClient() {
        if (httpClient == null) {
            httpClient = new HttpClient(
                new MultiThreadedHttpConnectionManager());
        }

        return httpClient;
    }

    /**
     * Returns a <code>HttpMethodBase</code> instance used by the
     * {@link #importData(String, String, Resource)} method to actually retrieve
     * the data to be imported from the data source URL.
     * <p>
     * This base class implementation returns a plain <code>GetMethod</code>.
     * Extensions may overwrite to implement more soffisticated method setup.
     * <p>
     * Each call to this method must return a new instance of the
     * <code>HttpMethodBase</code> implementation. The
     * {@link #importData(String, String, Resource)} method using this instance
     * takes care to release the connection of the method when being done.
     *
     * @param source The data source URL for which to prepare the method
     *            instance.
     * @return The <code>HttpMethodBase</code> to import the data, which is in
     *         this base class implementation a <code>GetMethod</code> instance.
     */
    protected HttpMethodBase newHttpMethod(String source) {
        return new GetMethod(source);
    }

    /**
     * Initializes this base class implementation. This method <b>must</b> be
     * called before the {@link #importData(String, String, Resource)} may be
     * used.
     * <p>
     * If the class is used as a Declarative Services component, this method
     * need not be called explicitly, since the
     * {@link #activate(ComponentContext)} method will call it.
     * <p>
     * This method may be called multiple times. After a first initialization
     * further successive calls without {@link #destroy() destroyal} will have
     * no effect.
     */
    protected final void init() {
        // nothing at the moment, might set configuration
    }

    /**
     * Destroys this base class implementation. This method <b>must</b> be
     * called when the instance is not used any more.
     * <p>
     * If the class is used as a Declarative Services component, this method
     * need not be called explicitly, since the
     * {@link #deactivate(ComponentContext)} method will call it.
     * <p>
     * This method may be called multiple times. After a first destroyal further
     * successive calls without {@link #init() initialization} will have no
     * effect.
     */
    protected final void destroy() {
        if (httpClient != null) {
            HttpConnectionManager cm = httpClient.getHttpConnectionManager();
            if (cm instanceof MultiThreadedHttpConnectionManager) {
                ((MultiThreadedHttpConnectionManager) cm).shutdown();
            }

            httpClient = null;
        }
    }

    // ---------- SCR integration

    /**
     * This base class implementation currently just calls the {@link #init()}
     * method. Nevertheless it <b>must</b> be called if overwritten!
     */
    protected void activate(ComponentContext context) {
        init();
    }

    /**
     * This base class implementation currently just calls the
     * {@link #destroy()} method. Nevertheless it <b>must</b> be called if
     * overwritten!
     */
    protected void deactivate(ComponentContext context) {
        destroy();
    }
}
