/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.day.cq.replication;

import org.apache.sling.api.resource.ValueMap;

/**
 * The configuration of a replication agent.
 */
public interface AgentConfig {

    /* basic agent params */
    String AGENT_NAME = "jcr:title";
    String AGENT_DESCRIPTION = "jcr:description";
    String AGENT_ENABLED = "enabled";
    String AGENT_LOG_LEVEL = "logLevel";
    String AGENT_USER_ID = "userId";
    /* trigger params */
    String TRIGGER_SPECIFIC = "triggerSpecific";
    String TRIGGER_MODIFIED = "triggerModified";
    String TRIGGER_ONOFFTIME = "triggerOnOffTime";
    String TRIGGER_DISTRIBUTE = "triggerDistribute";
    String REVERSE_REPLICATION = "reverseReplication";
    String TRIGGER_RECEIVE = "triggerReceive";
    /* queue params */
    String QUEUE_RETRY_DELAY = "retryDelay";
    String QUEUE_BATCH_MODE = "queueBatchMode";
    String QUEUE_BATCH_WAIT_TIME = "queueBatchWaitTime";
    String QUEUE_BATCH_MAX_SIZE = "queueBatchMaxSize";
    /**
     * @deprecated - this is not used anymore.
     */
    @Deprecated
    String QUEUE_MAX_RETRIES = "maxRetries";
    /* serialization params */
    String SERIALIZATION_TYPE = "serializationType";
    /* basic transport params */
    String TRANSPORT_URI = "transportUri";
    String TRANSPORT_USER = "transportUser";
    String TRANSPORT_PASSWORD = "transportPassword";
    /* extended authentication params */
    String TRANSPORT_NTLM_DOMAIN = "transportNTLMDomain";
    String TRANSPORT_NTLM_HOST = "transportNTLMHost";
    /* connection params */
    String PROTOCOL_VERSION = "protocolVersion";
    String PROTOCOL_CONNECT_TIMEOUT = "protocolConnectTimeout";
    String PROTOCOL_INTERFACE = "protocolInterface";
    String PROTOCOL_SOCKET_TIMEOUT = "protocolSocketTimeout";
    /* proxy params */
    String PROXY_HOST = "proxyHost";
    String PROXY_PORT = "proxyPort";
    String PROXY_USER = "proxyUser";
    String PROXY_PASS = "proxyPassword";
    String PROXY_NTLM_DOMAIN = "proxyNTLMDomain";
    String PROXY_NTLM_HOST = "proxyNTLMHost";
    /* http specific params */
    String PROTOCOL_HTTP_HEADERS = "protocolHTTPHeaders";
    String PROTOCOL_HTTP_METHOD = "protocolHTTPMethod";
    String PROTOCOL_HTTP_CONNECTION_CLOSE = "protocolHTTPConnectionClose";
    /* https specific params */
    String PROTOCOL_HTTPS_RELAXED = "protocolHTTPSRelaxed";
    String PROTOCOL_HTTPS_EXPIRED = "protocolHTTPExpired";
    /* OAuth 2.0 Bearer JWT as Authorization Grant */
    String ENABLE_OAUTH = "enableOauth";
    /* Access token provider pid */
    String ACCESS_TOKEN_PROVIDER_PID = "accessTokenProviderPid";

    /* SSL config settings */
    String SSL_CONFIG = "ssl";
    /**
     * Use default settings.
     */
    String SSL_CONFIG_DEFAULT = "default";
    /**
     * Use to accept self-certified SSL certificates.
     */
    String SSL_CONFIG_RELAXED = "relaxed";
    /**
     * Use to enable SSL client authentication.
     */
    String SSL_CONFIG_CLIENT_AUTH = "clientauth";
    /* distribute property */
    String CQ_DISTRIBUTE = "cq:distribute";
    /* extended params */
    String AGENT_NO_VERSIONS = "noVersioning";
    String AGENT_NO_STATUS_UPDATE = "noStatusUpdate";
    String AGENT_MAINTENANCE = "maintenanceMode";
    String AGENT_ALIAS_UPDATE = "aliasUpdate";

    /**
     * Checks if the configuration is valid and throws an exception if not. The message of the exception can be used as hint why the config
     * is not valid.
     *
     * @throws IllegalArgumentException if not valid.
     */
    public void checkValid();

    /**
     * Returns the id of this configuration.
     *
     * @return the id
     */
    String getId();

    /**
     * Returns the id for the agent of this configuration. note that there can be several configurations with the same agent id but in
     * different groups (i.e. run modes).
     *
     * @return the agent id
     */
    String getAgentId();

    /**
     * Returns the path to the configuration node. If the configuration is not stored in the repository, <code>null</code> is returned.
     *
     * @return the path to the config or <code>null</code>
     */
    String getConfigPath();

    /**
     * The descriptive name of this replication agent. This is a convenience method and returns the property {@link #AGENT_NAME}
     *
     * @return agent name
     */
    String getName();

    /**
     * Returns the group of this configuration
     *
     * @return the config group
     */
    AgentConfigGroup getConfigGroup();

    /**
     * The destination URI. This uri is one of those defined in the property {@link #TRANSPORT_URI}.
     *
     * @return the destination URI.
     */
    String getTransportURI();

    /**
     * The list of alternative destination URI.
     * This is a convenience method and returns the uris defined in the property {@link #TRANSPORT_URI}
     * as a comma separated list.
     *
     * @return the destination URI.
     */
    String[] getAllTransportURIs();


    /**
     * The target user name for delivering the content. This is a convenience method and returns the property {@link #TRANSPORT_USER}.
     *
     * @return transport user name
     */
    String getTransportUser();

    /**
     * The target user password for delivering the content. This is a convenience method and returns the property {@link
     * #TRANSPORT_PASSWORD}.
     *
     * @return plain text transport password
     */
    String getTransportPassword();

    /**
     * The name of the serialization type. A corresponding {@link ContentBuilder} with this name must be registered. This is a convenience
     * method and returns the property {@link #SERIALIZATION_TYPE}.
     *
     * @return name of the content builder
     */
    String getSerializationType();

    /**
     * The id of the user that is used to build the replication content. This is a convenience method and returns the property {@link
     * #AGENT_USER_ID}.
     *
     * @return the userid or <code>null</code>
     */
    String getAgentUserId();

    /**
     * The retry delay in milliseconds. This is a convenience method and returns the property {@link #QUEUE_RETRY_DELAY}.
     *
     * @return retry delay.
     */
    long getRetryDelay();

    /**
     * Checks if this agent is enabled This is a convenience method and returns the property {@link #AGENT_ENABLED}.
     *
     * @return <code>true</code> if this agent is enabled
     */
    boolean isEnabled();

    /**
     * The current log level - DEBUG, INFO, or ERROR. This is a convenience method and returns the property {@link #AGENT_LOG_LEVEL}.
     *
     * @return log lever
     */
    String getLogLevel();

    /**
     * The number of retries before the replication is cancelled. This is a convenience method and returns the property {@link
     * #QUEUE_MAX_RETRIES}.
     *
     * @return The number of retries or -1 for endless retries.
     * @deprecated This is handled by the queue configuration
     */
    @Deprecated
    int getMaxRetries();

    /**
     * Returns if the replication by this agent is performed in batch mode.
     *
     * @return If the replication is performed in batch mode.
     * @since 5.12.0
     */
    boolean isBatchMode();

    /**
     * If batch mode is enabled for this agent, this method returns a positive number in seconds. This time specifies the maximum wait time
     * for a batch to be sent.
     *
     * @return The batch wait time in seconds.
     * @since 5.12.0
     */
    long getBatchWaitTime();

    /**
     * If batch mode is enabled for this agent, this method returns a positive byte size. With the first replication exceeding this size,
     * the batch is sent.
     *
     * @return The maximum batch size in bytes.
     * @since 5.12.0
     */
    long getBatchMaxSize();

    /**
     * Checks if this agent is used for specific applications. Specific agents are used by the application for special application specific
     * replications and are not used during a general replication. This is a convenience method and returns the property {@link
     * #TRIGGER_SPECIFIC}.
     *
     * @return <code>true</code> if agent is specific
     */
    boolean isSpecific();

    /**
     * Checks if this agent is used for "replicate on modification". If this flag is <code>true</code> each modification to the content will
     * trigger a replication action on this agent. This is a convenience method and returns the property {@link #TRIGGER_MODIFIED}.
     *
     * @return <code>true</code> if this agent auto replicates on modification.
     */
    boolean isTriggeredOnModification();

    /**
     * Checks if this agent is used for "replicate if on-/offtime reached". If this flag is <code>true</code> any content that reached an
     * on-/offtime boundary will trigger a replication action on this agent. This is a convenience method and returns the property {@link
     * #TRIGGER_ONOFFTIME}.
     *
     * @return <code>true</code> if this agent auto replicates when on-/offtime reached
     */
    boolean isTriggeredOnOffTime();

    /**
     * Checks if this agent is used when a property "cq:distribute" has been modified. This is a convenience method and returns the property
     * {@link #TRIGGER_DISTRIBUTE}.
     *
     * @return <code>true</code> if this agent replicates on this property change
     */
    boolean isTriggeredOnDistribute();

    /**
     * Checks if this agent is used for reverse replication. This is a convenience method and returns the property {@link
     * #REVERSE_REPLICATION}.
     *
     * @return <code>true</code> if this agent is used for reverse replication.
     */
    boolean usedForReverseReplication();

    /**
     * <p>
     * Determines if a replication with this agent should suppress implicit versioning. If all participating agents of a replication specify
     * this flag, the respective flags are set in the replication options passed to the preprocessors.
     * </p>
     * <p>
     * This is a convenience method and returns the property {@link #AGENT_NO_VERSIONS}.
     * </p>
     * 
     * @return <code>true</code> if a replication with this agent should not trigger implicit versioning.
     * @since 5.4
     */
    boolean noVersions();

    /**
     * <p>
     * Determines if a replication with this agent should suppress update of the replication status. If all participating agents of a
     * replication specify this flag, the replication status is not updated after a replication.
     * </p>
     * <p>
     * This is a convenience method and returns the property {@link #AGENT_NO_STATUS_UPDATE}.
     * </p>
     * 
     * @return <code>true</code> if a replication with this agent should not update the replication status.
     * @since 5.4
     */
    boolean noStatusUpdate();

    /**
     * Returns All agent properties as value map.
     *
     * @return agent properties
     */
    ValueMap getProperties();

    /**
     * Determines if a replication with this agent should be triggered once the replication request is received. This is used particularly
     * for chain replication
     *
     * @return <code>true</code> if the agent should be triggered once replication is received
     * @since 5.5.19
     */
    boolean isTriggeredOnReceive();

    /**
     * Checks if this agent is in maintenance mode This is a convenience method and returns the property {@link #AGENT_MAINTENANCE}.
     *
     * @return <code>true</code> if this agent is in maintenance mode
     */
    boolean isInMaintenanceMode();

    /**
     * Determines if a replication with this agent should also replicate aliases, vanity URLs.
     *
     * @return <code>true</code> if agent should handle replication for aliases, vanity URLs
     */
    boolean aliasUpdate();

    /**
     * Returns SSL config mode. Values are default, relaxed or clientauth.
     *
     * @see AgentConfig#SSL_CONFIG
     */
    String getSSLConfig();

    /**
     * Checks whether the agent allows the use of expired certificates or not.
     *
     * @return {@code true} if the agent accepts expired certificates, {@code false} otherwise
     */
    boolean allowsExpiredCertificates();

    /**
     * Checks whether the agent is using the OAuth 2.0 Authorization Grants mechanism or not. This is a convenience method
     * and returns the property {@link #ENABLE_OAUTH}
     *
     * @return {@code true} if the authorization grants mechanism is enabled
     */
    boolean isOAuthEnabled();
}
