/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.search.eval;

import javax.jcr.Session;
import javax.jcr.query.Row;

import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.cq.search.Predicate;
import com.day.cq.search.PredicateGroup;
import com.day.cq.search.Query;
import com.day.cq.search.QueryBuilder;

/**
 * <code>SavedQueryPredicate</code> includes all predicates of a saved query
 * into the current query. No extra query is run, instead the final xpath query
 * expression will contain the merge of the all predicates of the current query
 * plus all predicates from the saved query.
 * 
 * <p>
 * Please note that there won't be any facet extraction happening for the
 * predicates from the saved query. To do so, you can write custom code that
 * first {@linkplain QueryBuilder#loadQuery(String, Session) loads the query}
 * and executes it directly.
 * 
 * <h3>Name:</h3>
 * savedquery
 *
 * <h3>Parameters:</h3>
 * <dl>
 * <dt>savedquery</dt>
 * <dd>path to the saved query</dd>
 * <dl>
 *
 * @since 5.3
 */
@Component(metatype = false, factory = "com.day.cq.search.eval.PredicateEvaluator/savedquery")
public class SavedQueryPredicate extends PredicateGroupEvaluator {
    
    private static final Logger log = LoggerFactory.getLogger(SavedQueryPredicate.class);
    
    public static final String SAVED_QUERY = "savedquery";
    
    @Reference
    public QueryBuilder queryBuilder;
    
    @Override
    public String getXPathExpression(Predicate p, EvaluationContext context) {
        PredicateGroup group = getSavedPredicates(p, context);
        if (group != null) {
            return super.getXPathExpression(group, context);
        }
        // no saved query specified or found => no xpath
        return null;
    }
    
    @Override
    public boolean includes(Predicate p, Row row, EvaluationContext context) {
        PredicateGroup group = getSavedPredicates(p, context);
        if (group != null) {
            return super.includes(group, row, context);
        }
        // no saved query specified or found => no filtering
        return true;
    }

    @Override
    public boolean canXpath(Predicate p, EvaluationContext context) {
        PredicateGroup group = getSavedPredicates(p, context);
        if (group != null) {
            return super.canXpath(group, context);
        }
        // no saved query specified or found => no xpath
        return false;
    }

    @Override
    public boolean canFilter(Predicate p, EvaluationContext context) {
        PredicateGroup group = getSavedPredicates(p, context);
        if (group != null) {
            return super.canFilter(group, context);
        }
        // no saved query specified or found => no filtering
        return false;
    }
    
    // ---------------------------------------------------------------< private >

    protected String getEvalContextKey(Predicate p) {
        return p.getPath() + ".savedquery";
    }
    
    protected PredicateGroup getSavedPredicates(Predicate p, EvaluationContext context) {
        if (!p.hasNonEmptyValue(SAVED_QUERY)) {
            return null;
        }
        
        String key = getEvalContextKey(p);
        PredicateGroup group = null;
        Object obj = context.get(key);
        if (obj == null) {
            // first time during this evaluation, try to load
            group = loadPredicates(p.get(SAVED_QUERY), context.getSession());
            if (group == null) {
                // set dummy marker to avoid re-loading every time
                context.put(key, new Object());
            } else {
                // persist loaded predicates in evaluation context for later calls
                context.put(key, group);
            }
        } else if (obj instanceof PredicateGroup) {
            group = (PredicateGroup) obj;
        }
        return group;
    }

    protected PredicateGroup loadPredicates(String path, Session session) {
        try {
            // load the predicates from the saved query
            Query query = queryBuilder.loadQuery(path, session);
            if (query == null) {
                // not found
                log.warn("No saved query found at '" + path + "'");
                return null;
            }
            return query.getPredicates();
            
        } catch (Exception e) {
            log.error("Could not load query from '" + path + "'", e);
            return null;
        }
    }
    
}
