/*
 * Copyright 1997-2010 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.search.eval;

import org.apache.jackrabbit.util.ISO9075;
import org.apache.jackrabbit.util.Text;

/**
 * Utility class for XPath related methods and constants.
 *
 */
public abstract class XPath {
    
    public static final String SEARCH_ALL = "//*";
    
    public final static String NOT = "not";
    
    public final static String AND = " and ";
    
    public final static String OR = " or ";
    
    public final static String ORDER_BY = " order by ";
    
    public final static String DESC = " descending";
    
    public final static String OPENING_BRACKET = "(";
    
    public final static String CLOSING_BRACKET = ")";
    
    public final static String PREDICATE_OPENING_BRACKET = "[";
    
    public final static String PREDICATE_CLOSING_BRACKET = "]";
    
    public final static String JCR_ROOT = "/jcr:root";
    
    public final static String JCR_LIKE = "jcr:like";
    
    public final static String JCR_CONTAINS = "jcr:contains";

    public final static String JCR_LIKE_WILDCARD = "%";

    public final static char   JCR_LIKE_ANY_WILDCARD = '%';

    public final static char   JCR_LIKE_SINGLE_WILDCARD = '_';

    public final static String REP_EXCERPT = "/rep:excerpt(.)";

    public static final String FN_LOWER_CASE = "fn:lower-case";

    public static String getPropertyPath(String property) {
        if (property.charAt(0) == '@') {
            property = property.substring(1);
        }
        
        String name = Text.getName(property);
        if (name.charAt(0) != '@') {
            name = '@' + ISO9075.encode(name);
        } else {
            name = '@' + ISO9075.encode(name.substring(1));
        }
        
        String parent = Text.getRelativeParent(property, 1);
        if (parent.length() > 0) {
            return ISO9075.encodePath(parent) + "/" + name;
        } else {
            return name;
        }
    }
    
    public static String getStringLiteral(String value) {
        return "'" + value.replaceAll("'", "''") + "'";
    }
    
    private static int count(String s, char c) {
        if (s == null || s.length() == 0) {
            return 0;
        }
        int count = 0;
        for (int i=0; i < s.length(); i++) {
            if (s.charAt(i) == c) {
                count++;
            }
        }
        return count;
    }
    
    public static String getFulltextStringLiteral(String value) {
        // double-quotes are used to build search term groups
        // if an uneven number occurs (eg. one), Jackrabbit's Lucene search parser chokes,
        // thus we have to escape the last double-quote by putting a backslash in front
        int doubleQuotes = count(value, '"');
        if (doubleQuotes % 2 == 1) {
            int last = value.lastIndexOf('"');
            value = value.substring(0, last) + "\\" + value.substring(last);
        }
        // escape special full text search chars (at the end of the string)
        value = Text.escapeIllegalXpathSearchChars(value);
        // put in single-quotes and double escape them inside the string
        return "'" + value.replaceAll("'", "''") + "'";
    }
    
    public static String getEqualsExpression(String property, String value) {
        return getPropertyPath(property) + " = " + getStringLiteral(value);
    }
    
    public static String getUnequalsExpression(String property, String value) {
        return getPropertyPath(property) + " != " + getStringLiteral(value);
    }
    
    public static String getJcrLikeExpression(String property, String value) {
        return JCR_LIKE + "(" + XPath.getPropertyPath(property) + ", " + getStringLiteral(value) + ")";
    }
    
    public static String getNotExpression(String property) {
        return NOT + "(" + XPath.getPropertyPath(property) + ")";
    }

    public static String getXPathOrderBy(String property, boolean ascending) {
        return getXPathOrderBy(property, ascending, false);
    }

    public static String getXPathOrderBy(String property, boolean ascending, boolean ignoreCase) {
        // orderby for single property : orderby @jcr:lastModified
        // orderby for path to property: orderby jcr:content/@jcr:lastModified
        if (ignoreCase) {
            return FN_LOWER_CASE + OPENING_BRACKET + getPropertyPath(property) + CLOSING_BRACKET + (ascending ? "" : DESC);
        } else {
            return getPropertyPath(property) + (ascending ? "" : DESC);
        }
    }
}
