/*
 * Copyright 1997-2009 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.search.writer;

import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

import javax.jcr.RepositoryException;

import org.apache.felix.scr.annotations.Component;
import org.apache.jackrabbit.util.Text;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceMetadata;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.commons.json.JSONException;
import org.apache.sling.commons.json.io.JSONWriter;

import com.day.cq.search.Query;
import com.day.cq.search.result.Hit;

/**
 * {@linkplain SimpleHitWriter} writes common properties that are available on
 * most JCR nodes, eg. path, excerpt, name, title, lastModified, created, size
 * and mimeType.
 */
@Component(metatype = false, factory = "com.day.cq.search.writer.ResultHitWriter/simple")
public class SimpleHitWriter implements ResultHitWriter {

    public void write(Hit hit, JSONWriter writer, Query query) throws RepositoryException, JSONException {
        writeSimpleJson(hit, writer);
    }
    
    public final static long KILO_BYTE = 1024;
    public final static long MEGA_BYTE = 1024 * KILO_BYTE;
    public final static long GIGA_BYTE = 1024 * MEGA_BYTE;
    public final static long TERA_BYTE = 1024 * GIGA_BYTE;
    public final static long PETA_BYTE = 1024 * TERA_BYTE;
    
    public static void writeSimpleJson(Hit hit, JSONWriter writer) throws RepositoryException, JSONException {
        // Note: we are not accessing the javax.jcr.Node or properties here, only using
        // Resource and the valuemap returned by the Hit (which points to jcr:content, if present)
        
        final Resource resource = hit.getResource();
        final ResourceMetadata metadata = resource.getResourceMetadata();
        final ValueMap properties = hit.getProperties();
        
        writer.key("path").value(hit.getPath());
        
        final String excerpt = hit.getExcerpt();
        if (excerpt != null) {
            writer.key("excerpt").value(excerpt);
        }
        
        final String name = Text.getName(hit.getPath());
        writer.key("name").value(name);
        writer.key("title").value(properties.get("jcr:title", name));
        
        final SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        // 1. try resource metadata
        long lastModified = metadata.getModificationTime();
        if (lastModified < 0) {
            // 2. try cq:lastModified (pages)
            // 3. try jcr:lastModified
            Calendar cal = (Calendar) properties.get("cq:lastModified");
            if (cal == null) {
                cal = (Calendar) properties.get("jcr:lastModified");
            }
            if (cal != null) {
                lastModified = cal.getTimeInMillis();
            }
        }
        if (lastModified >= 0) {
            writer.key("lastModified").value(format.format(new Date(lastModified)));
        }
        // 1. try resource metadata
        long created = metadata.getCreationTime();
        if (created < 0) {
            // 2. try jcr:created
            Calendar cal = (Calendar) properties.get("jcr:created");
            if (cal != null) {
                created = cal.getTimeInMillis();
            }
        }
        if (created >= 0) {
            writer.key("created").value(format.format(new Date(created)));
        }
        
        long size = metadata.getContentLength();
        if (size < 0) {
            // resource metadata is only set if inputstream was fetched
            resource.adaptTo(InputStream.class);
            size = metadata.getContentLength();
        }
        if (size >= 0) {
            String humanSize;
            if (size >= PETA_BYTE) {
                humanSize = (size / PETA_BYTE) + " PB";
            } else if (size >= TERA_BYTE) {
                humanSize = (size / TERA_BYTE) + " TB";
            } else if (size >= GIGA_BYTE) {
                humanSize = (size / GIGA_BYTE) + " GB";
            } else if (size >= MEGA_BYTE) {
                humanSize = (size / MEGA_BYTE) + " MB";
            } else if (size >= KILO_BYTE) {
                humanSize = (size / KILO_BYTE) + " KB";
            } else {
                humanSize = size + " Bytes";
            }
            writer.key("size").value(humanSize);
        }
        
        String mimeType = metadata.getContentType();
        if (mimeType == null) {
            mimeType = (String) properties.get("jcr:mimeType");
        }
        if (mimeType != null) {
            writer.key("mimeType").value(mimeType);
        }
    }

}
