/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.tagging;

import java.util.Iterator;
import java.util.Locale;
import java.util.Map;

import javax.jcr.Node;

import aQute.bnd.annotation.ProviderType;
import org.apache.sling.api.adapter.Adaptable;
import org.apache.sling.api.resource.Resource;

import com.day.cq.commons.Filter;

/**
 * A <code>Tag</code> is a keyword or label that can be assigned to any
 * content. Multiple tags can be assigned to a content element, which allows to
 * classify them by several different categorizations at the same time, as
 * opposed to a single hierarchical model. Tags became popular with the wave of
 * Web 2.0 websites, but the CQ5 tagging model goes even further in flexibility
 * by offering both <em>namespaces</em>, <em>hierarchical tags</em> and the
 * ability to <em>restrict users to use only predefined tags</em>.
 * 
 * <h3>Retrieving a Tag object</h3>
 * A Tag can be retrieved through the {@link TagManager}, by either resolving
 * an existing tag or creating a new one:
 * 
 * <pre><code>
 * Tag tag = tagManger.resolve(&quot;my/tag&quot;); // for existing tags
 * 
 * Tag tag = tagManger.createTag(&quot;my/tag&quot;); // for new tags
 * </code></pre>
 * 
 * For the {@linkplain JcrTagManagerFactory JCR-based implementation}, which
 * maps tags onto JCR {@link Node}s, you can directly use Sling's apaptTo
 * mechanism if you happen to have the resource (eg. such as
 * <code>/etc/tags/default/my/tag</code>):
 * <pre><code>
 * Tag tag = resource.adaptTo(Tag.class);
 * </code></pre>
 * (Note: adapting from {@link Node} to a tag is not possible directly, since
 * {@link Node} does not implement the {@link Adaptable#adaptTo(Class)} method).
 * <br>
 * These conversions are vice-versa, you can get the node and resource back from a tag
 * by calling:
 * <pre><code>
 * Node node = tag.adaptTo(Node.class);
 * 
 * Resource node = tag.adaptTo(Resource.class);
 * </code></pre>
 * 
 * <h3>Overview of Tagging concepts</h3>
 * A <b>tag</b> can be a simple word (eg. <code>beautiful</code>) or
 * represent a hierarchical taxonomy (eg. <code>planet/earth</code>, meaning
 * both the generic <code>planet</code> and the more specific
 * <code>earth</code>). Tags are typically identified by strings, either by a
 * so-called <em>short
 * tag ID</em> (typical case) or an
 * <em>absolute path to the tag</em> (a rather rare case). A tag has meta
 * information such as a title (that should be displayed in UIs instead of the
 * tag ID, if present) and a description.
 * 
 * <p>
 * Furthermore, <b>container tags</b> (or super-tags) in a taxonomy serve as
 * the sub-summation of all sub-tags, eg. content tagged with
 * <code>fruit/banana</code> is considered to be tagged with <code>fruit</code> as
 * well, ie. searching for content just tagged with <code>fruit</code> would
 * find the ones with <code>fruit/banana</code> as well.
 * 
 * <p>
 * Finally, <b>namespaces</b> allow to group things. The most typical use-case
 * is to have a namespace per (web)site (eg. public, internal, portal, etc.) or
 * for larger applications (eg. wcm, dam, collab) - but you are free to use
 * namespaces for whatever you need. Namespaces are used in the UI to only show
 * the subset of tags (ie. tags of a certain namespace) that is applicable for
 * the current content.
 * 
 * <h3>Tagging Terminology</h3>
 * <p>
 * A <b>tag ID</b> is either a shorthand or an absolute path refering to a tag.
 * It must be resolved to an absolute <b>tag path</b>. A tag is contained
 * within a certain <b>namespace</b> (which is <code>default</code> if no
 * namespace is given in the tag ID). The part after the namespace is called the
 * <b>local tag ID</b> and can be a path itself. <b>container tags</b> have
 * sub-tags that represent a hierarchical order in the taxonomy. They can be
 * used as <b>tags</b> (an actually referenced tag) like any other tag below a
 * namespace (eg. tagging something with <code>planet</code> is not forbidden
 * if it has sub-tags like <code>planet/earth</code> and
 * <code>planet/mars</code> for example).
 * 
 * <h3>Examples of Tag IDs</h3>
 * <h4>Fully qualified tag ID: <code>dam:fruit/apple/braeburn</code></h4>
 * <dl>
 * <dt>namespace</dt>
 * <dd><code>dam</code></dd>
 * <dt>tag local ID</dt>
 * <dd><code>fruit/apple/braeburn</code></dd>
 * <dt>container tags</dt>
 * <dd><code>fruit</code> and <code>apple</code></dd>
 * <dt>leaf tag</dt>
 * <dd><code>braeburn</code></dd>
 * <dt>absolute tag path</dt>
 * <dd><code>/etc/tags/dam/fruit/apple/braeburn</code></dd>
 * </dl>
 * 
 * <h4>Shorter example: <code>color/red</code></h4>
 * <dl>
 * <dt>namespace</dt>
 * <dd><code>default</code></dd>
 * <dt>tag local ID</dt>
 * <dd><code>color/red</code></dd>
 * <dt>container tag</dt>
 * <dd><code>color</code></dd>
 * <dt>leaf tag</dt>
 * <dd><code>red</code></dd>
 * <dt>absolute tag path</dt>
 * <dd><code>/etc/tags/default/color/red</code></dd>
 * </dl>
 * 
 * <h4>Simplest example: <code>sky</code></h4>
 * <dl>
 * <dt>namespace</dt>
 * <dd><code>default</code></dd>
 * <dt>tag local ID</dt>
 * <dd><code>sky</code></dd>
 * <dt>container tag</dt>
 * <dd>(none)</dd>
 * <dt>leaf tag</dt>
 * <dd><code>sky</code></dd>
 * <dt>absolute tag path</dt>
 * <dd><code>/etc/tags/default/sky</code></dd>
 * </dl>
 * 
 * <h4>Namespace example: <code>dam:</code></h4>
 * <dl>
 * <dt>namespace</dt>
 * <dd><code>dam</code></dd>
 * <dt>tag local ID</dt>
 * <dd>(none)</dd>
 * <dt>container tag</dt>
 * <dd>(none)</dd>
 * <dt>leaf tag</dt>
 * <dd>(none, the namespace)</dd>
 * <dt>absolute tag path</dt>
 * <dd><code>/etc/tags/dam</code></dd>
 * </dl>
 */
@ProviderType
public interface Tag extends Adaptable {
    
    /**
     * Returns the node name, ie. either the name of the namespace if this Tag
     * represents a tag namespace, the name of the part of a taxonomy (eg. "fruit"
     * in "fruit/apple") or the name of the tag itself.
     */
    String getName();
    
    /**
     * Returns the short tag ID. If namespace is <code>default</code>, it
     * will be left out. If this tag represents a namespace, <code>null</code>
     * is returned.
     */
    String getTagID();

    /**
     * Returns the tag path below a namespace. If this tag represents a
     * namespace, <code>null</code> is returned.
     */
    String getLocalTagID();
    
    /**
     * Returns the full absolute path of a tag or a tag namespace.
     */
    String getPath();
    
    /**
     * Returns a title that should be displayed instead of the tag name or tag
     * id. If no separate title is available, the tag name is returned.
     */
    String getTitle();

    /**
     * Returns a localized title for the given locale, if available. Given a
     * locale such as "de_CH", this will return in this order based on
     * availability:
     * 
     * <ol>
     * <li>country-specific title "de_CH"</li>
     * <li>language title "de"</li>
     * <li>default tag title</li>
     * </ol>
     * 
     * If locale is <code>null</code>, this will return the same as
     * {@link #getTitle()}. If you only want localized titles (and
     * <code>null</code> if no localization exists), use
     * {@link #getLocalizedTitle(Locale)}.
     */
    String getTitle(Locale locale);

    /**
     * Returns a localized title for the given locale, if available. Given a
     * locale such as "de_CH", this will return in this order based on
     * availability:
     * 
     * <ol>
     * <li>country-specific title "de_CH"</li>
     * <li>language title "de"</li>
     * <li><code>null</code></li>
     * </ol>
     * 
     * If no localization is found, this returns <code>null</code>. If a
     * fallback to the default {@link #getTitle()} is desired, use
     * {@link #getTitle(Locale)} instead.
     */
    String getLocalizedTitle(Locale locale);
    
    /**
     * Returns all available localized titles.
     * 
     * @see #getLocalizedTitle(Locale)
     */
    Map<Locale, String> getLocalizedTitles();
    
    /**
     * Returns an optional description for a tag. If none is available,
     * <code>null</code> is returned.
     */
    String getDescription();
    
    /**
     * Returns the title variant of the tag ID: a path made up of the single
     * titles of the namespace and all tags leading to this tag.
     */
    String getTitlePath();

    /**
     * Returns the localized title variant of the tag ID: a path made up of the
     * single titles of the namespace and all tags leading to this tag.
     * 
     * Given a locale such as "de_CH", this will return in this order based on
     * availability:
     * 
     * <ol>
     * <li>country-specific title path "de_CH"</li>
     * <li>language title path "de"</li>
     * <li>default tag title path</li>
     * </ol>
     * 
     * If locale is <code>null</code>, this will return the same as
     * {@link #getTitlePath()}.
     */
    String getTitlePath(Locale locale);

    /**
     * Returns all available localized title paths.
     * 
     * @see #getTitlePath(Locale)
     */
    Map<Locale, String> getLocalizedTitlePaths();
    
    /**
     * Returns the usage count of the tag, which can be used for tag cloud and
     * other visualisations. An implementation does not have to ensure a completely
     * exact number, so caching is possible.
     */
    long getCount();
    
    /**
     * Returns the date of last modification of the tag.
     *
     * @return utc millis of the last modification date
     */
    long getLastModified();

    /**
     * Return the user id that last modified the Tag or <code>null</code> if this
     * information is not available.
     * @return user id or <code>null</code>
     */
    String getLastModifiedBy();

    /**
     * Returns <code>true</code> if this tag object refers to a tag namespace.
     */
    boolean isNamespace();

    /**
     * Returns the tag namespace inside which this tag resides as tag object.
     */
    Tag getNamespace();
    
    /**
     * Gets the parent of this tag. This could be either a tag container, a tag
     * namespace or <code>null</code> if this object is a tag namespace.
     */
    Tag getParent();
    
    /**
     * Lists the direct child tags for both tag namespaces and tag containers.
     */
    Iterator<Tag> listChildren();
    
    /**
     * Lists the child tags for both tag namespaces and tag containers, but only
     * those included by a filter.
     */
    Iterator<Tag> listChildren(Filter<Tag> filter);
    
    /**
     * Lists all sub-tags, ie. children, grand-children and so on. Note that the
     * returned list might not be in order.
     */
    Iterator<Tag> listAllSubTags();
    
    /**
     * Returns all nodes tagged with this tag. These are all nodes with a
     * <code>cq:tags</code> property that contains either the tagID or the
     * full absolute path to that tag. Returns <code>null</code> in case of
     * errors.
     * 
     * @see TagManager#find(String)
     * @see TagManager#find(String, String[])
     */
    Iterator<Resource> find();

    /**
     * Returns the jcr xpath query constraint for searching this tag. This will
     * return something like
     * <code>relative/path/@cq:tags = "tag:id" or relative/path/@cq:tags = "tag:previous/id"</code>
     * considering child tags and different tag ids because of moved or merged tags.
     *
     * @param property relative path to {@link TagConstants#PN_TAGS cq:tags} property,
     *                     e.g. <code>cq:tags</code> or <code>relative/path/@cq:tags</code>
     *
     * @return an xpath expression to be used inside a constraint (<code>//*[...]</code>)
     */
    String getXPathSearchExpression(String property);

    /**
     * Returns the GQL query constraint for searching this tag. This will
     * return something like
     * <code>"cq:tags":"c" OR "cq:tags":"/etc/tags/default/c" OR "cq:tags":"c/b" OR "cq:tags":"/etc/tags/default/c/b"</code>
     * considering child tags and different tag ids because of moved or merged tags.
     *
     * @param property relative path to {@link TagConstants#PN_TAGS cq:tags} property,
     *                     e.g. <code>cq:tags</code> or <code>relative/path/@cq:tags</code>
     *
     * @return a GQL (org.apache.jackrabbit.commons.query.GQL) expression
     */
    String getGQLSearchExpression(String property);
}
