/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.api;

import java.util.Collection;
import java.util.Locale;
import java.util.Map;

import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.Resource;

import com.day.cq.commons.Language;
import com.day.cq.commons.jcr.JcrConstants;

/**
 * Provides a service for content languages. Note that this has nothing to do
 * with the i18n of the WCM itself.
 */
public interface LanguageManager {

    /**
     * relative path of the language property of a hierarchy node.
     */
    static final String ISO_PROP_NAME = NameConstants.NN_CONTENT + "/" + JcrConstants.JCR_LANGUAGE;

    /**
     * Returns the country iso code for the given locale. if the country is not
     * already defined in the locale a configured default value is returned.
     * If no iso code exists, an empty string is returned.
     *
     * @param language the language locale to retrieve the country from
     * @return a country iso code or ""
     */
    String getIsoCountry(Locale language);

    /**
     * Returns a map of language information for each
     * language filled with information if respective hierarchy or content exists
     * for that language.
     *
     * @param resolver resource resolver
     * @param path path to check
     * @return map of information or <code>null</code> if the given path does not
     *         reside on or below a language root.
     *
     * @see com.day.cq.commons.LanguageUtil#getLanguageRoot(String)
     *
     * @deprecated use {@link #getAdjacentLanguageInfo(ResourceResolver, String)} instead
     */
    Map<Locale, Info> getAdjacentInfo(ResourceResolver resolver, String path);

    /**
     * Returns a map of language information for each
     * language filled with information if respective hierarchy or content exists
     * for that language.
     *
     * @param resolver resource resolver
     * @param path path to check
     * @return map of information or <code>null</code> if the given path does not
     *         reside on or below a language root.
     *
     * @see com.day.cq.commons.LanguageUtil#getLanguageRoot(String)
     *
     * @since 5.4
     */
    Map<Language, Info> getAdjacentLanguageInfo(ResourceResolver resolver, String path);

    /**
     * Returns the language for the given resource.
     * Note that this method only uses the path names for the determination of
     * the language.
     *
     * @param res resource
     * @return the locale or <code>null</code> if not determinable
     */
    Locale getLanguage(Resource res);

    /**
     * Returns the language for the given resource.
     * Note that this method only uses the path names for the determination of
     * the language.
     *
     * @param res resource
     * @return the locale or <code>null</code> if not determinable
     *
     * @since 5.4
     */
    Language getCqLanguage(Resource res);

    /**
     * Returns the language for the given resource. The path of the resource
     * is analyzed to find a matching iso code. If no iso code can be found,
     * <code>null</code> is returned unless <code>respectContent</code> is
     * <code>true</code>. then the resources ancestors (including the given)
     * one is searched for a {@link #ISO_PROP_NAME} property.
     *
     * @param res resource
     * @param respectContent if <code>false</code> only the path is analyzes
     * @return the locale or <code>null</code> if not determinable
     *
     * @since 5.4
     */
    Locale getLanguage(Resource res, boolean respectContent);

    /**
     * Returns the language for the given resource. The path of the resource
     * is analyzed to find a matching iso code. If no iso code can be found,
     * <code>null</code> is returned unless <code>respectContent</code> is
     * <code>true</code>. then the resources ancestors (including the given)
     * one is searched for a {@link #ISO_PROP_NAME} property.
     *
     * @param res resource
     * @param respectContent if <code>false</code> only the path is analyzes
     * @return the locale or <code>null</code> if not determinable
     *
     * @since 5.4
     */
    Language getCqLanguage(Resource res, boolean respectContent);

    /**
     * Returns the language root page for the given resource.
     * Note that currently only the path names are respected for the
     * determination of the language.
     * @param res resource
     * @return the language root page or <code>null</code> if not determinable
     */
    Page getLanguageRoot(Resource res);

    /**
     * Returns a collection of root languages for the given page. Note that
     * only the path names are respected for the determination of the language.
     *
     * @param resolver resource resolver
     * @param path path of the current page
     * @return collection of root languages
     *
     * @deprecated use {@link #getCqLanguages(ResourceResolver, String)} instead.
     */
    Collection<Locale> getLanguages(ResourceResolver resolver, String path);

    /**
     * Returns a collection of root languages for the given page. Note that
     * only the path names are respected for the determination of the language.
     *
     * @param resolver resource resolver
     * @param path path of the current page
     * @return collection of root languages
     *
     * @since 5.4
     */
    Collection<Language> getCqLanguages(ResourceResolver resolver, String path);

    /**
     * Returns a collection of language root pages for the given page. Note that
     * only the path names are respected for the determination of the language.
     *
     * @param resolver resource resolver
     * @param path path of the current page
     * @return collection of language root paths
     */
    Collection<Page> getLanguageRoots(ResourceResolver resolver, String path);

    /**
     * Compares language trees using the language of the page at the given path
     * as main language.
     * @param resolver resource resolved
     * @param path path to resource to start from
     * @return the tree information
     */
    public Tree compareLanguageTrees(ResourceResolver resolver, String path);

    /**
     * Encapsulates information about language specific existences of resources
     * in a tree.
     */
    interface Tree {

        /**
         * Returns the root path of the tree
         * @return the root path of the tree
         */
        String getRoot();

        /**
         * Returns a collection of all languages covered by this tree
         * @return collection of languages
         *
         * @deprecated use {@link #getCqLanguages()} instead
         */
        Collection<Locale> getLanguages();

        /**
         * Returns a collection of all languages covered by this tree
         * @return collection of languages
         *
         * @since 5.4
         */
        Collection<Language> getCqLanguages();

        /**
         * Returns an collection of nodes.
         * @return the nodes
         */
        Collection<? extends TreeNode> getNodes();

    }

    /**
     * Denotes a node in the tree.
     */
    interface TreeNode {

        /**
         * Returns the relative path of the node.
         * @return the relative path
         */
        String getRelativePath();

        /**
         * Returns the info for the given locale or <code>null</code> if the
         * locale is not part of the language collection returned by
         * {@link Tree#getLanguages()}.
         *
         * @param locale the locale
         * @return info or <code>null</code>
         *
         * @deprecated use {@link #getInfo(Language)} instead
         */
        Info getInfo(Locale locale);

        /**
         * Returns the info for the given locale or <code>null</code> if the
         * locale is not part of the language collection returned by
         * {@link Tree#getCqLanguages()}.
         *
         * @param locale the locale
         * @return info or <code>null</code>
         *
         * @since 5.4
         */
        Info getInfo(Language locale);

    }

    /**
     * Information about a resource in a language tree.
     */
    interface Info {

        /**
         * Returns the path
         * @return the path
         */
        String getPath();

        /**
         * Checks if the resource exists at the given path.
         * @return <code>true</code> if the resource exists at the given path
         */
        boolean exists();

        /**
         * Checks if the resource has content at the given path
         * @return <code>true</code> if the resource has content at the given path
         */
        boolean hasContent();

        /**
         * Returns the last modification date of the resource
         * @return the last modification date or 0
         */
        long getLastModified();

    }

}