/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.api;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.sling.event.dea.DEAConstants;
import org.osgi.service.event.Event;

/**
 * This event is fired for each page change.
 */
public final class PageEvent implements Serializable {

    /**
     * Event topic for page modification events.
     */
    public static final String EVENT_TOPIC = "com/day/cq/wcm/core/page";

    /**
     * event property name of the modifications
     */
    private static final String PROPERTY_MODIFICATIONS = "modifications";

    private static final String PROPERTY_APPLICATION = "event.application";
    private static final String PROPERTY_DISTRIBUTE = "event.distribute";

    /**
     * Is this a local event?
     */
    private final boolean isLocal;

    /**
     * The list of modifications
     */
    private final List<PageModification> modifications;

    public PageEvent(PageModification mod) {
        this(Collections.singletonList(mod), true);
    }

    public PageEvent(final List<PageModification> mods) {
        this(mods, true);
    }

    public PageEvent(final List<PageModification> mods, final boolean isLocal) {
        this.modifications = mods;
        this.isLocal = isLocal;
    }

    /**
     * Get the list of modifications.
     *
     * @return An iterator for the modifications.
     */
    public Iterator<PageModification> getModifications() {
        return this.modifications.iterator();
    }

    /**
     * Is this a local or a cluster event?
     * @return <code>true</code> if this is a local event
     */
    public boolean isLocal() {
        return this.isLocal;
    }

    /**
     * Convert an OSGi event to a page event if possible.
     *
     * @param evt The OSGi event.
     * @return The page even or null.
     */
    public static PageEvent fromEvent(Event evt) {
        if (!evt.getTopic().equals(EVENT_TOPIC)) {
            return null;
        }
        final List<PageModification> mods = new ArrayList<PageModification>();
        @SuppressWarnings("unchecked")
        final List<Map<String, Object>> modProps = (List<Map<String, Object>>) evt.getProperty(PROPERTY_MODIFICATIONS);
        // sanity check for null
        if (modProps != null) {
            for (Map<String, Object> modProp: modProps) {
                mods.add(PageModification.fromEventProperties(modProp));
            }
        }

        return new PageEvent(mods, (evt.getProperty(DEAConstants.PROPERTY_APPLICATION) == null ? true : false) );
    }

    /**
     * Generate a dictionary for the OSGi event.
     *
     * @return an event property map
     */
    protected Dictionary<String, Object> getEventProperties() {
        //using dictionary here, so that implementation of toEvent does not need to
        // convert from map to dictionary
        final Dictionary<String, Object> properties = new Hashtable<String, Object>();
        final List<Map<String, Object>> modProps = new ArrayList<Map<String, Object>>();
        final Iterator<PageModification> i = this.getModifications();
        while (i.hasNext()) {
            modProps.add(i.next().getEventProperties());
        }
        properties.put(PROPERTY_MODIFICATIONS, modProps);
        if (!isLocal) {
            properties.put(PROPERTY_APPLICATION, "unknown");
        }

        return properties;
    }

    /**
     * Create a distributable event.
     *
     * @return An event.
     */
    public Event toEvent() {
        Dictionary<String, Object> props = this.getEventProperties();
        // for now the value has no meaning, so we just put an empty string in it.
        props.put(PROPERTY_DISTRIBUTE, "");
        return new Event(EVENT_TOPIC, props );

    }

    /**
     * Create a non distributable event.
     *
     * @return An event.
     */
    public Event toNonDistributableEvent() {
        return new Event(EVENT_TOPIC, this.getEventProperties());
    }
}
