/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.api;

import javax.servlet.ServletRequest;

import com.day.cq.commons.Filter;

/**
 * Implements a simple page filter that checks for {@link Page#isValid() invalid}
 * and {@link Page#isHideInNav() hidden} pages.
 */
public class PageFilter implements Filter<Page> {

    /**
     * controls if invalid pages should be included
     */
    private final boolean includeInvalid;

    /**
     * controls if hidden should be included
     */
    private final boolean includeHidden;

    /**
     * creates a default filter that excludes invalid and hidden pages.
     */
    public PageFilter() {
        this(false, false);
    }

    /**
     * creates a default filter that excludes hidden pages. invalid pages are
     * only excluded if the WCM mode is {@link WCMMode#DISABLED}.
     * @param mode the WCM mode
     */
    public PageFilter(WCMMode mode) {
        this(mode != WCMMode.DISABLED, false);
    }

    /**
     * creates a default filter that excludes hidden pages. invalid pages are
     * only excluded if the WCM mode is {@link WCMMode#DISABLED}.
     * @param req request to read the WCM mode from
     */
    public PageFilter(ServletRequest req) {
        this(WCMMode.fromRequest(req));
    }

    /**
     * creates a filter.
     *
     * @param includeInvalid if <code>true</code> invalid pages are included.
     * @param includeHidden if <code>true</code> hidden pages are included.
     */
    public PageFilter(boolean includeInvalid, boolean includeHidden) {
        this.includeInvalid = includeInvalid;
        this.includeHidden = includeHidden;
    }

    /**
     * {@inheritDoc}
     */
    public boolean includes(Page page) {
        return !(!includeHidden && page.isHideInNav())
                && !(!includeInvalid && !page.isValid())
                && (page.getDeleted() == null); // add override flag later "includeDelete"
    }

    /**
     * Checks if invalid pages are included.
     * @return <code>true</code> if invalid pages are included.
     */
    public boolean isIncludeInvalid() {
        return includeInvalid;
    }

    /**
     * Checks if hidden pages are included
     * @return <code>true</code> if hidden pages are included.
     */
    public boolean isIncludeHidden() {
        return includeHidden;
    }

}