/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */

package com.day.cq.wcm.api.designer;

import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.api.NameConstants;
import com.day.cq.wcm.api.PageManager;
import org.apache.sling.api.resource.Resource;

/**
 * Specifies the interface of the designer. It provides methods for retrieving
 * design information.
 * Designs are stored below {@value #DESIGN_HOME} and mirror the contents
 * structure to some extend. The pages hold path references to the designs.
 * If not path reference is specified or not inherited, the default design at
 * {@value #DEFAULT_DESIGN_PATH} is used.
 */
public interface Designer {

    /**
     * Home of the designs
     */
    String DESIGN_HOME = "/etc/designs";

    /**
     * path to default design
     */
    String DEFAULT_DESIGN_PATH = "/etc/designs/default";

    /**
     * node type for design nodes. use pages for now.
     */
    String NT_DESIGN = NameConstants.NT_PAGE;

    /**
     * Returns the path to the design of the given page. If it does not specify
     * such a path, the page's parent design path is returned or
     * {@link #DEFAULT_DESIGN_PATH} if the page does not have a parent.
     *
     * @param page the page
     * @return the design path or <code>null</code> if <code>page</code> is <code>null</code>;
     */
    String getDesignPath(Page page);

    /**
     * Returns the design for the given page.
     * @param page the page to retrieve the design from.
     * @return the design or <code>null</code> if the <code>page</code> is <code>null</code>.
     */
    Design getDesign(Page page);

    /**
     * Checks if the design with the given id exists.
     * @param id the design id
     * @return true if the design exists.
     */
    boolean hasDesign(String id);

    /**
     * Returns a design by id.
     * @param id id of the design
     * @return the design or the default design if it does not exist.
     * @see Design#getId()
     */
    Design getDesign(String id);

    /**
     * Returns the style for a given resource and cell path. It's a convenience
     * method that retrieves the containing page of the resource, resolves
     * its design and then gets the style. if <code>cellPath</code> is
     * <code>null</code> the name of the resource is used.
     *
     * @param res the resource
     * @param cellPath the path of the design cell
     * @return the style or <code>null</code>
     * @see PageManager#getContainingPage(Resource)
     * @see #getDesign(Page)
     * @see Design#getStyle(String)
     */
    Style getStyle(Resource res, String cellPath);

    /**
     * Returns the style for a given resource. It's a convenience
     * method that retrieves the containing page of the resource, resolves
     * its design and then gets the style. the cell path is constructed by walking
     * the content path.
     *
     * @param res the resource
     * @return the style or <code>null</code>
     * @see PageManager#getContainingPage(Resource)
     * @see #getDesign(Page)
     * @see Design#getStyle(String)
     */
    Style getStyle(Resource res);
}