/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.commons;

import java.io.UnsupportedEncodingException;
import java.util.BitSet;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.Calendar;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.PropertyType;
import javax.jcr.RepositoryException;
import javax.jcr.Value;
import javax.jcr.query.Query;

import org.apache.jackrabbit.util.Text;
import org.apache.sling.api.SlingException;
import org.apache.sling.api.resource.PersistenceException;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceUtil;
import org.apache.sling.jcr.api.SlingRepository;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.cq.commons.predicate.AbstractResourcePredicate;
import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.api.PageManager;
import com.day.cq.wcm.api.WCMException;

/**
 * <code>ReferenceSearch</code> provides methods that search references to
 * resources (e.g. a path in a property to an image)
 */
public class ReferenceSearch {

    private SlingRepository repository;

    /**
     * default logger
     */
    private static final Logger log = LoggerFactory.getLogger(ReferenceSearch.class);

    /**
     * search root
     */
    private String searchRoot = "/";

    /**
     * exact
     */
    private boolean exact = false;

    /**
     * hollow
     */
    private boolean hollow = false;

    /**
     * reference limit per page
     */
    private int maxReferencesPerPage = -1;
    
    /**
     * Resource predicate, can be set to filter results based on a resource predicate {@link AbstractResourcePredicate}
     */
    private AbstractResourcePredicate resourcePredicate = null;

    /**
     * @deprecated The repository was used by the implementation to create an administrative session in
     * {@link #adjustReferences(ResourceResolver, String, String, String[])}. Instead, specify a resource resolver
     * which is sufficiently authorized to adjust references at the desired locations.
     */
    public ReferenceSearch setRepository(SlingRepository repository) {
        this.repository = repository;
        return this;
    }

    /**
     * Returns the search root. default is '/'
     * @return the search root.
     */
    public String getSearchRoot() {
        return searchRoot;
    }

    /**
     * Sets the search root
     * @param searchRoot the search root
     * @return this
     */
    public ReferenceSearch setSearchRoot(String searchRoot) {
        if (searchRoot == null || searchRoot.equals("")) {
            this.searchRoot = "/";
        } else {
            this.searchRoot = searchRoot;
        }
        return this;
    }

    /**
     * Returns the shallow flag.
     * @return the shallow flag.
     * @see #setExact(boolean)
     */
    public boolean isExact() {
        return exact;
    }

    /**
     * Sets the <code>exact</code> flag. If <code>true</code> only exact
     * references are searched (and replaced). otherwise also references to
     * child resources are included.
     *
     * @param exact <code>true</code> if perform an exact search
     * @return this;
     */
    public ReferenceSearch setExact(boolean exact) {
        this.exact = exact;
        return this;
    }

    /**
     * Returns the <code>hollow</code> flag.
     * @return the <code>hollow</code> flag.
     * @see #setHollow(boolean)
     */
    public boolean isHollow() {
        return hollow;
    }

    /**
     * Sets the <code>hollow</code> flag. If <code>true</code>
     * the returned info will contain only properties of the page
     * and not the page object itself.
     * @param hollow <code>true</code> if perform a hollow search
     * @return this;
     */
    public ReferenceSearch setHollow(boolean hollow) {
        this.hollow = hollow;
        return this;
    }

    /**
     * Returns the maximum number of references that can be added to a page info.
     * @return the reference limit per page.
     * @see #setMaxReferencesPerPage(int)
     */
    public int getMaxReferencesPerPage() {
        return maxReferencesPerPage;
    }

    /**
     * Sets the value of <code>maxReferencesPerPage</code>.
     * The maximum number of references that can be added to a page info.
     * @param maxReferencesPerPage
     * @return this;
     */
    public ReferenceSearch setMaxReferencesPerPage(int maxReferencesPerPage) {
        this.maxReferencesPerPage = maxReferencesPerPage;
        return this;
    }
    
    /**
     * Set the <code>resourcePredicate</code>, this predicate is evaluated for
     * each search result and can be used to filter results.
     * 
     * @param resourcePredicate an instance of {@link AbstractResourcePredicate}
     * @return this;
     */
    public ReferenceSearch setPredicate(AbstractResourcePredicate resourcePredicate) {
        this.resourcePredicate = resourcePredicate;
        return this;
    }

    /**
     * Searches for references to the given path.
     * @param resolver the resource resolver
     * @param path the path to search for
     * @return reference infos
     */
    public Map<String, Info> search(ResourceResolver resolver, String path) {
        if (path == null) {
            return Collections.emptyMap();
        }
        String root = searchRoot.equals("/") ? "" : searchRoot;
        PageManager manager = resolver.adaptTo(PageManager.class);
        Map<String, Info> infos = new HashMap<String, Info>();

        Pattern pattern = getSearchPattern(path);
        String qPath = escapeIllegalXpathSearchChars(path);
        String query = String.format("%s//*[jcr:contains(., '\"%s\"')]", root, qPath);
        search(resolver, manager, infos, pattern, query);

        // also search for escaped path, if contains special characters
        String escPath = Text.escapePath(path);
        if (!escPath.equals(path)) {

            // already rewritten values: all hex lower case encoded
            Pattern escPattern = getSearchPattern(escPath);
            String qEscPath = escapeIllegalXpathSearchChars(escPath);
            query = String.format("%s//*[jcr:contains(., '%s')]", root, qEscPath);
            search(resolver, manager, infos, escPattern, query);

            // RTE in Classic UI and in Touch UI post CQ-4241224: all hex upper case encoded
            escPath = escapePathUsingUpperCaseHex(path);
            escPattern = getSearchPattern(escPath);
            qEscPath = escapeIllegalXpathSearchChars(escPath);
            query = String.format("%s//*[jcr:contains(., '%s')]", root, qEscPath);
            search(resolver, manager, infos, escPattern, query);

            // RTE in Touch UI pre CQ-4241224: & encoded as HTML entity; others hex upper case encoded
            if (escPath.contains("%26")) {
                escPath = escPath.replaceAll("%26", "&amp;");
                escPattern = getSearchPattern(escPath);
                qEscPath = escapeIllegalXpathSearchChars(escPath);
                query = String.format("%s//*[jcr:contains(., '%s')]", root, qEscPath);
                search(resolver, manager, infos, escPattern, query);
            }
        }

        // filter out those infos that are empty
        for (Iterator<Map.Entry<String, Info>> entries = infos.entrySet().iterator(); entries.hasNext();) {
            Map.Entry<String, Info> entry = entries.next();
            if (entry.getValue().getProperties().isEmpty()) {
                entries.remove();
            } else {
                //filter based on predicate
                if (resourcePredicate != null) {
                    if (entry.getValue().page != null) {
                        Resource pageResource = entry.getValue().page.adaptTo(Resource.class);
                        if (pageResource != null) {
                            if (!resourcePredicate.evaluate(pageResource)) {
                                entries.remove();
                            }
                        }
                    }
                }
            }
        }
        return infos;
    }

    private String escapePathUsingUpperCaseHex(String string) {
        try {
            BitSet validChars = Text.URISaveEx;
            char escape = '%';
            final char[] hexTable = "0123456789ABCDEF".toCharArray();
            byte[] bytes = string.getBytes("utf-8");
            StringBuilder out = new StringBuilder(bytes.length);
            for (byte aByte : bytes) {
                int c = aByte & 0xff;
                if (validChars.get(c) && c != escape) {
                    out.append((char) c);
                } else {
                    out.append(escape);
                    out.append(hexTable[(c >> 4) & 0x0f]);
                    out.append(hexTable[(c) & 0x0f]);
                }
            }
            return out.toString();
        } catch (UnsupportedEncodingException e) {
            throw new InternalError(e.toString());
        }
    }

    private void search(ResourceResolver resolver,
                        PageManager manager, Map<String, Info> infos,
                        Pattern pattern, String query) {
        log.debug("Searching for references using: {}", query);
        Iterator<Resource> iter = null;
        try{
            iter = resolver.findResources(query, Query.XPATH);
        }catch(SlingException e){
            log.warn("error finding resources", e);
            return;
        }

        // process the search results and build the result set
        while (iter.hasNext()) {
            Resource res = iter.next();
            Page page = manager.getContainingPage(res);
            if (page != null) {
                Info info = infos.get(page.getPath());
                if (info == null) {
                    info = new Info(page, hollow);
                    infos.put(page.getPath(), info);
                }
                try {
                    // analyze the properties of the resource
                    Node node = res.adaptTo(Node.class);
                    for (PropertyIterator pIter = node.getProperties(); pIter.hasNext();) {
                        // don't add properties any further if limit is exceeded
                        if (getMaxReferencesPerPage() >= 0 && info.getProperties().size() >= getMaxReferencesPerPage()) {
                            break;
                        }
                        Property p = pIter.nextProperty();
                        // only check string and name properties
                        if (p.getType() == PropertyType.STRING || p.getType() == PropertyType.NAME) {
                            if (p.isMultiple()) {
                                for (Value v: p.getValues()) {
                                    String value = v.getString();
                                    if (pattern.matcher(value).find()) {
                                        info.addProperty(p.getPath());
                                        break;
                                    }
                                }
                            } else {
                                String value = p.getString();
                                if (pattern.matcher(value).find()) {
                                    info.addProperty(p.getPath());
                                }
                            }
                        }
                    }
                } catch (RepositoryException e) {
                    log.error("Error while accessing " + res.getPath(), e);
                }
            }
        }
    }

    /**
     * <p>Adjusts all references to <code>path</code> to <code>destination</code>
     * in the pages specified by <code>refPaths</code>. If {@link #isExact()}
     * is <code>true</code> only exact references to <code>path</code> are
     * adjusted, otherwise all references to child resources are adjusted, too.</p>
     *
     * <p>The resource resolver needs to have sufficient permissions (i.e. <code>jcr:read</code> and
     * <code>rep:alterProperties</code>) on the nodes containing references.</p>
     *
     * @param resolver resolver to operate on.
     * @param path source path
     * @param destination destination path
     * @param refPaths paths of pages to be adjusted
     * @return collection of path to properties that were adjusted
     */
    public Collection<String> adjustReferences(ResourceResolver resolver,
                                               String path, String destination,
                                               String[] refPaths) {
        if (refPaths == null) {
            return Collections.emptyList();
        }
        Set<String> adjusted = new HashSet<String>();
        for (String p: refPaths) {
            Resource r = resolver.getResource(p);
            if (r == null) {
                log.warn("Given path does not address a resource: {}", p);
                continue;
            }
            Page page = r.adaptTo(Page.class);
            if (page == null) {
                log.warn("Given path does not address a page: {}", p);                
            }
            
            Resource content = page != null ? page.getContentResource() : null;           
            if (content == null) {
                log.warn("Given page does not have content: {}", p);
            }
                       
            try {
                //Can be a case of complex asset.
                Node node = content != null ? content.adaptTo(Node.class) : r.adaptTo(Node.class);
                adjusted.addAll(adjustReferences(node, path, destination));

                // CQ5-32249 - touch the pages outside of this loop to avoid an inefficient O(n^2) algorithm
            } catch (RepositoryException e) {
                log.error("Error while adjusting references on " + r.getPath(), e);
            }

            // #22466 - moving pages does not take into account usergenerated content
            try {
                String adjustedUGCPath = adjustUserGeneratedContentReference(r, path, destination);
                if (adjustedUGCPath != null) {
                    adjusted.add(adjustedUGCPath);
                    log.info("Adjusted user generated content path {}.", adjustedUGCPath);
                }
            } catch (Exception e) {
                log.error("Error while adjusting user generated references on " + r.getPath(), e);
            }
        }

        // CQ5-32249 - do the touch calls after the above loop to avoid an inefficient O(n^2) algorithm
        PageManager pm = resolver.adaptTo(PageManager.class);
        // #38440 - touch the pages that were adjusted
        for (final String pathOfAdjusted : adjusted) {
            final Resource adjustedResource = resolver.getResource(pathOfAdjusted);
            if (null != adjustedResource) {
                final Page adjustedPage = pm.getContainingPage(adjustedResource);
                if (null != adjustedPage) {
                    try {
                        pm.touch(adjustedPage.adaptTo(Node.class), true, Calendar.getInstance(), false);
                    } catch (WCMException e) {
                        log.error("could not update last modified on adjusted page [{}]: ", adjustedPage.getPath(), e);
                    }
                }
            }
        }

        // save changes
        try {
            resolver.commit();
        } catch (PersistenceException e) {
            log.error("Error while adjusting references.", e);
        }

        return adjusted;
    }

    /**
     * Adjusts all references to <code>path</code> to <code>destination</code>
     * in the properties below the specified <code>node</code>. If {@link #isExact()}
     * is <code>true</code> only exact references to <code>path</code> are
     * adjusted, otherwise all references to child resources are adjusted, too.
     *
     * @param node (content) node to traverse
     * @param path source path
     * @param destination destination path
     * @throws RepositoryException if an error during repository access occurs
     * @return collection of paths to properties that were adjusted
     */
    public Collection<String> adjustReferences(Node node, String path, String destination)
            throws RepositoryException {
        return adjustReferences(node, path, destination, false, Collections.<String>emptySet());
    }

    /**
     * Adjusts all references to <code>path</code> to <code>destination</code>
     * in the properties below the specified <code>node</code>. If {@link #isExact()}
     * is <code>true</code> only exact references to <code>path</code> are
     * adjusted, otherwise all references to child resources are adjusted, too.
     *
     * @param node (content) node to adjust
     * @param path source path
     * @param destination destination path
     * @param shallow if <code>true</code> child nodes are not traversed
     * @param excludedProperties a set of excluded property names
     * @throws RepositoryException if an error during repository access occurs
     * @return collection of paths to properties that were adjusted
     */
    public Collection<String> adjustReferences(Node node, String path,
                                               String destination, boolean shallow,
                                               Set<String> excludedProperties)
            throws RepositoryException {
        Set<String> adjusted = new HashSet<String>();
        Pattern pattern = getReplacementPattern(path);
        String escDest = Text.escapePath(destination);
        for (PropertyIterator iter = node.getProperties(); iter.hasNext();) {
            Property p = iter.nextProperty();
            // only check string and name properties
            if (!excludedProperties.contains(p.getName()) &&
                    p.getType() == PropertyType.STRING || p.getType() == PropertyType.NAME) {
                if (p.isMultiple()) {
                    Value[] values = p.getValues();
                    boolean modified = false;
                    for (int i=0; i<values.length; i++) {
                        String value = rewrite(values[i].getString(), path, pattern, destination, escDest);
                        if (value != null) {
                            values[i] = node.getSession().getValueFactory().createValue(
                                    value, p.getType());
                            modified = true;
                        }
                    }
                    if (modified) {
                        p.setValue(values);
                        adjusted.add(p.getPath());
                        log.info("Adjusted property {}.", p.getPath());
                    }
                } else {
                    String value = rewrite(p.getString(), path, pattern, destination, escDest);
                    if (value != null) {
                        p.setValue(value);
                        adjusted.add(p.getPath());
                        log.info("Adjusted property {}.", p.getPath());
                    }
                }
            }
        }
        // traverse child nodes
        if (!shallow) {
            for (NodeIterator iter = node.getNodes(); iter.hasNext();) {
                adjusted.addAll(adjustReferences(iter.nextNode(), path, destination,shallow, excludedProperties));
            }
        }
        return adjusted;
    }

    /**
     * Adjust user generated content reference
     *
     * @param resource reference to adjust
     * @param path source path
     * @param destination destination path
     * @return the new path of the user generated content reference
     * @throws RepositoryException if an error during repository access occurs
     * @throws WCMException if an error during UGC page move occurs
     */
    private String adjustUserGeneratedContentReference(Resource resource, String path, String destination) throws RepositoryException, WCMException {
        // Check if the resource that has been adjusted maps to a user generated content resource
        String ugcPath = resource.getPath();
        if (ugcPath.startsWith(Constants.PATH_UGC + path)) {
            // Determine the new UGC path
            String newUgcPath = ugcPath.replaceFirst(path, destination);

            // Check that the new UGC content resource does not exist yet
            ResourceResolver resolver = resource.getResourceResolver();
            Resource newUgcResource = resolver.resolve(newUgcPath);
            if (ResourceUtil.isNonExistingResource(newUgcResource)) {
                PageManager pageManager = resolver.adaptTo(PageManager.class);

                // Prepare user generated content until the parent of the page we will move
                String newUgcParentPath = Text.getRelativeParent(UGCUtil.UGCToResourcePath(newUgcResource), 1);
                UGCUtil.prepareUserGeneratedContent(resolver, newUgcParentPath);

                // Move the page
                pageManager.move(resource, newUgcPath, Text.getName(resource.getPath()), true, true, new String[0]);

                return newUgcPath;
            }
        }

        return null;
    }

    /**
     * Returns the search pattern
     * @param path source path
     * @return search pattern
     */
    protected Pattern getSearchPattern(String path) {
        if (exact) {
            return Pattern.compile("([\"']|^)(" + Pattern.quote(path) + ")([;.?#\"']|$)");
        } else {
            return Pattern.compile("([\"']|^)(" + Pattern.quote(path) + ")([;.?#\"'/](.*)|$)");
        }
    }

    /**
     * Returns the replacement pattern for the rewrite method. this pattern
     * matches only links in (single) quotes
     * @param path source path
     * @return replacement pattern
     */
    protected Pattern getReplacementPattern(String path) {
        String escPath = Text.escapePath(path);
        String literal = Pattern.quote(path);
        if (!escPath.equals(path)) {
            // already rewritten values: all hex lower case encoded
            literal += "|" + Pattern.quote(escPath);

            // RTE in Classic UI and in Touch UI post CQ-4241224: all hex upper case encoded
            escPath = escapePathUsingUpperCaseHex(path);
            literal += "|" + Pattern.quote(escPath);

            // RTE in Touch UI pre CQ-4241224: & encoded as HTML entity; others hex upper case encoded
            if (escPath.contains("%26")) {
                literal += "|" + Pattern.quote(escPath.replaceAll("%26", "&amp;"));
            }
        }
        if (exact) {
            return Pattern.compile("([\"'])(" + literal + ")([;.?#\"'])");
        } else {
            return Pattern.compile("([\"'])(" + literal + ")([;.?#\"'/])");
        }
    }

    /**
     * Internal rewrite method.
     * @param value the property value
     * @param from original path
     * @param p the replacement pattern
     * @param to to path
     * @param escTo escaped to path
     * @return rewritten path or <code>null</code> if not matches
     */
    protected String rewrite(String value, String from, Pattern p, String to, String escTo) {
        // first check unescaped direct property value
        if (value.equals(from)) {
            return to;
        } else if (value.startsWith(from + "#")
                || value.startsWith(from + ".html")) {
            // #34356 - handle cases where the path is followed by
            // an anchor or the .html suffix
            // TODO: There should be a less brittle way of doing this!
            return to + value.substring(from.length());
        } else if (!exact) {
            if (value.startsWith(from + "/")) {
                return to + value.substring(from.length());
            }
        }
        // ... then replace escaped references in rich text properties
        Matcher m = p.matcher(value);
        StringBuffer ret = null;
        String repl = "$1" + escTo + "$3";
        while (m.find()) {
            if (ret == null) {
                ret = new StringBuffer();
            }
            m.appendReplacement(ret, repl);
        }
        if (ret == null) {
            return null;
        } else {
            m.appendTail(ret);
            return ret.toString();
        }
    }

    /**
     * Escapes illegal XPath search characters.
     *
     * @param s the string to encode
     * @return the escaped string
     */
    public static String escapeIllegalXpathSearchChars(String s) {
        StringBuffer sb = new StringBuffer();
        for (char c: s.toCharArray()) {
            if (c == '!' || c == '(' || c == ')' || c == ':' || c == '^'
                || c == '[' || c == ']' || c == '{' || c == '}' || c == '?'
                || c == '"' || c == '\\' || c == ' ' || c == '~') {
                sb.append('\\');
            } else if (c == '\'') {
                sb.append(c);
            }
            sb.append(c);
        }
        return sb.toString();
    }

    /**
     * Holds information about the search results
     */
    public static final class Info {

        private final Page page;
        private final String pageTitle;
        private final String pagePath;

        private final Set<String> properties = new HashSet<String>();

        public Info(Page page) {
            this.page = page;
            pageTitle = page.getTitle();
            pagePath = page.getPath();
        }

        public Info(Page page, boolean hollow) {
            if (!hollow) {
                this.page = page;
            } else {
                this.page = null;
            }
            pageTitle = page.getTitle();
            pagePath = page.getPath();
        }

        private void addProperty(String path) {
            properties.add(path);
        }

        public Page getPage() {
            return page;
        }

        public Set<String> getProperties() {
            return properties;
        }

        public String getPageTitle() {
            return pageTitle;
        }

        public String getPagePath() {
            return pagePath;
        }
    }

}