/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.commons;

import java.util.Locale;

import javax.jcr.Node;
import javax.servlet.ServletRequest;

import com.adobe.cq.launches.api.Launch;
import com.day.cq.commons.inherit.InheritanceValueMap;
import com.day.cq.tagging.Tag;
import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.api.PageManager;
import com.day.cq.wcm.api.components.Component;
import com.day.cq.wcm.api.components.ComponentContext;
import com.day.cq.wcm.api.components.ComponentManager;
import com.day.cq.wcm.api.designer.Design;
import com.day.cq.wcm.api.designer.Designer;
import com.day.cq.wcm.api.designer.Style;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The <code>CMUtils</code> class provides utility methods for various
 * aspects of the Core WCM system.
 */
public class WCMUtils {

    /**
     * default logger
     */
    private static final Logger log = LoggerFactory.getLogger(WCMUtils.class);

    /**
     * debug parameter name
     */
    public static final String DEBUG_PARAM = "debug";

    // not to be instantiated
    private WCMUtils() {
    }

    /**
     * Returns the component according to the type of the resource or
     * <code>null</code> if no such component exists. This is a pure convenience
     * method retrieving the component manager and invoking
     * {@link ComponentManager#getComponentOfResource(Resource)}.
     *
     * @param resource to resolve
     * @return the respective component or <code>null</code>
     */
    public static Component getComponent(Resource resource) {
        if (resource == null) {
            log.debug("Resource to get component from must not be null");
            return null;
        }
        ComponentManager compMgr = resource.getResourceResolver().adaptTo(ComponentManager.class);
        if (compMgr == null) {
            log.warn("Unable to retrieve component manager for {}", resource);
            return null;
        }
        return compMgr.getComponentOfResource(resource);
    }

    /**
     * Returns the content property of the given page. If the page does not
     * contain this property, the page's parent's property is returned or
     * <code>null</code> if the page has no parent.
     * 
     * <p>
     * Note that the standard "pageProperties" variable defined by the
     * <code>&lt;cq:defineObjects/&gt;</code> tag is now (since CQ 5.4) an
     * {@link InheritanceValueMap} that provides
     * {@link InheritanceValueMap#getInherited(String, Class)
     * getInherited(String, Class)} and
     * {@link InheritanceValueMap#getInherited(String, Object)
     * getInherited(String, Object)} to search for those properties in parent
     * pages if not found locally.
     * 
     * @param page
     *            the page to retrieve the property from
     * @param resolver
     *            for resolving parent pages
     * @param name
     *            the property name
     * @return the property value or <code>null</code>
     */
    public static String getInheritedProperty(Page page,
                                              ResourceResolver resolver,
                                              String name) {
        Page previousPage;
        String value = null;
        while (value == null && page != null) {
            value = page.getProperties().get(name, String.class);
            if (value == null) {
                Resource res = resolver.getResource(page.getPath() + "/..");
                if (res == null) {
                    page = null;
                } else {
                    previousPage = page;
                    page = res.adaptTo(Page.class);
                    if (page == null) {
                        Launch launch = previousPage.adaptTo(Resource.class).adaptTo(Launch.class);
                        if (launch != null) {
                            page = launch.getSourceRootResource().adaptTo(Page.class);
                        }
                    }
                }
            }
        }
        return value;
    }

    /**
     * Returns the style for the given request. It's a convenience method
     * that retrieves the respective page of the request resource, resolves its
     * design, extracts the cell path and retrieves the style.
     *
     * A design and cell path can be specified '/-/' delimited as suffix.
     * the design path only needs to include the part after the /etc/designs.
     * eg: ...img.png/geometrixx/-/par/image
     *
     * @param request the request
     * @return the style or <code>null</code>
     */
    public static Style getStyle(SlingHttpServletRequest request) {
        // try to figure out the 'current' page. which is either the one
        // that is rendered, or the one that contains the requested resource
        ComponentContext ctx = getComponentContext(request);
        Resource res = request.getResource();
        Designer d = request.getResourceResolver().adaptTo(Designer.class);
        String designId = "";
        String cellPath = "";

        // grab paths from suffix, if present
        String suffix = request.getRequestPathInfo().getSuffix();
        if (suffix != null && suffix.length() > 0) {
            if (!suffix.startsWith("/")) {
                suffix = "/" + suffix;
            }
            int idx = suffix.indexOf("/-/");
            if (idx >= 0) {
                designId = suffix.substring(0, idx);
                cellPath = suffix.substring(idx + 3);
            } else {
                designId = suffix;
            }
        }

        // get design
        Design design;
        if (designId.length() > 0 && d.hasDesign(designId)) {
            design = d.getDesign(designId);
        } else {
            Page page = ctx == null ? null : ctx.getPage();
            if (page == null) {
                PageManager pMgr = request.getResourceResolver().adaptTo(PageManager.class);
                page = pMgr.getContainingPage(res);
            }
            cellPath = "";
            design = d.getDesign(page);
        }
        if (design == null) {
            return null;
        }

        // get style
        if (cellPath.length() > 0) {
            return design.getStyle(cellPath);
        } else if (ctx == null) {
            return design.getStyle(res);
        } else {
            return design.getStyle(ctx.getCell());
        }
    }

    /**
     * Returns the component context for the given request or <code>null</code>
     * if not defined.
     *
     * @param request servlet request
     * @return component context or <code>null</code>
     */
    public static ComponentContext getComponentContext(ServletRequest request) {
        return (ComponentContext) request.getAttribute(ComponentContext.CONTEXT_ATTR_NAME);
    }

    /**
     * Returns a node for the given resource. If the node does not
     * exist, returns null
     *
     * @param resource resource to adapt to node
     * @return the node or <code>null</code>
     */
    public static Node getNode(Resource resource) {
        if(resource!=null) {
            return resource.adaptTo(Node.class);
        }
        return null;
    }

    /**
     * Returns a comma-separated list of keywords for the given page, based on
     * the titles of the tags set on the page. Useful for the &lt;meta
     * keywords&gt; element in HTML.
     * 
     * <p>
     * Since CQ 5.4, this returns localized tag titles based on the page
     * language or the default title if no localized title exists for that
     * language. This is essentially the same as
     * {@link #getKeywords(Page, boolean) getKeywords(page, false)}.
     * 
     * @param page
     *            wcm page
     * @return comma-separated list of keywords
     */
    public static String getKeywords(Page page) {
        return getKeywords(page, false);
    }

    /**
     * Returns a comma-separated list of keywords for the given page, based on
     * the titles of the tags set on the page. Useful for the &lt;meta
     * keywords&gt; element in HTML.
     * 
     * <p>
     * Returns localized tag titles based on the page language. A boolean flag
     * allows to choose if only localized titles shall be returned or if it is
     * ok to fallback to the default title in case no localized title exists for
     * a given tag.
     * 
     * @since 5.4
     * 
     * @param page
     *            wcm page
     * @param onlyLocalized
     *            if only keywords should be returned that have a localized
     *            variant for the given page locale
     * @return comma-separated list of keywords
     */
   public static String getKeywords(Page page, boolean onlyLocalized) {
        StringBuffer keywords = new StringBuffer();
        if (page != null) {
            Locale locale = page.getLanguage(false);
            Tag[] tags = page.getTags();
            for (int i=0; i < tags.length; i++) {
                if (onlyLocalized) {
                    String title = tags[i].getLocalizedTitle(locale);
                    if (title != null) {
                        if (keywords.length() > 0) keywords.append(", ");
                        keywords.append(title);
                    }
                } else {
                    if (keywords.length() > 0) keywords.append(", ");
                    keywords.append(tags[i].getTitle(locale));
                }
            }
        }
        return keywords.toString();
    }
}