/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.core.contentfinder;

import com.day.cq.commons.feed.StringResponseWrapper;
import com.day.cq.tagging.Tag;
import com.day.cq.tagging.TagManager;
import org.apache.commons.collections.Predicate;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.SlingHttpServletResponse;
import org.apache.sling.api.request.RequestDispatcherOptions;
import org.apache.sling.api.request.RequestPathInfo;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.commons.json.JSONArray;
import org.apache.sling.commons.json.JSONException;
import org.apache.sling.commons.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.Session;
import javax.servlet.RequestDispatcher;
import javax.servlet.ServletException;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.Locale;

public abstract class ViewHandler extends ContentFinderListInfoProviderHelper {
    private static final long serialVersionUID = 5964360462202342062L;

    private static final Logger LOG = LoggerFactory.getLogger(ViewHandler.class);

    /** Query clause */
    public static final String QUERY = "query";

    /** type clause */
    public static final String TYPE = "type";

    /** limit clause */
    public static final String LIMIT = "limit";

    /** default limit for result sets */
    public static final int DEFAULT_LIMIT = 20;

    public static final String ITEM_RESOURCE_TYPE = "itemResourceType";

    // ----------------------< Servlet Methods >--------------------------------

    /**
     * @see com.day.cq.commons.servlets.AbstractPredicateServlet#doGet(org.apache.sling.api.SlingHttpServletRequest,
     *      org.apache.sling.api.SlingHttpServletResponse, org.apache.commons.collections.Predicate)
     */
    @Override
    protected void doGet(SlingHttpServletRequest request, SlingHttpServletResponse response, Predicate predicate) throws ServletException,
            IOException {
        JSONArray hits = new JSONArray();
        JSONObject result = new JSONObject();
        RequestPathInfo pathInfo = request.getRequestPathInfo();
        Collection<Hit> results = executeSearch(request);
        if ("json".equals(pathInfo.getExtension())) {
            response.setContentType("application/json");
            response.setCharacterEncoding("utf-8");

            try {
                hits = getHitsArray(results, request, null);
                result.put("hits", hits);
            } catch (Exception e) {
                throw new ServletException(e);
            } finally {
                response.getWriter().write(result.toString());
            }
        } else {
            // html output
            response.setContentType("text/html");
            response.setCharacterEncoding("utf-8");

            try {
                response.getWriter().write(generateHtmlOutput(results, request, response));
            } catch (Exception e) {
                throw new ServletException(e);
            }
        }
    }

    protected Collection<Hit> executeSearch(SlingHttpServletRequest request) throws ServletException {
        Collection<Hit> results = null;

        String queryString = request.getParameter(QUERY);
        if (queryString == null || queryString.length() == 0) {
            queryString = "";
        }
        queryString = queryString.trim();

        try {
            ResourceResolver resolver = request.getResourceResolver();
            Session session = resolver.adaptTo(Session.class);
            ViewQuery q = createQuery(request, session, queryString);
            results = q.execute();
        } catch (Exception e) {
            throw new ServletException(e);
        }
        return results;
    }

    protected abstract ViewQuery createQuery(SlingHttpServletRequest request, Session session, String queryString) throws Exception;

    protected JSONArray getHitsArray(Iterable<Hit> hits, SlingHttpServletRequest request, Resource resource) throws JSONException {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        JSONArray hitsArray = new JSONArray();
        for (Hit hit : hits) {
            JSONObject jsonHit = new JSONObject();
            Iterator<String> keys = hit.getKeys();
            while (keys.hasNext()) {
                String key = keys.next();
                Object val = hit.get(key);
                if (val instanceof Calendar) {
                    Date date = ((Calendar) val).getTime();
                    jsonHit.put(key, format.format(date));
                } else {
                    jsonHit.put(key, val);
                }
            }
            String path = null;
            try {
                boolean shouldResetResource = false;
                if (resource == null) {
                    if ((path = (String) jsonHit.get("path")) != null) {
                        ResourceResolver rr = request.getResourceResolver();
                        resource = rr.getResource(path);
                        if (resource != null) {
                            shouldResetResource = true;
                        }
                    }
                }
                if (resource != null) {
                    pingCallbacksWithItem(request, jsonHit, resource);
                    if (shouldResetResource) {
                        resource = null;
                    }
                }
            } catch (JSONException e) {
                LOG.warn("Could not find a path element in the JSON object; not calling ListInfoProviders");
            }
            hitsArray.put(jsonHit);
        }
        return hitsArray;
    }

    /**
     * Generate html output for Servlet
     */
    protected String generateHtmlOutput(Collection<Hit> hits, SlingHttpServletRequest request,
                                      SlingHttpServletResponse response) throws ServletException, IOException {
        // if resource type is not set return String representation of search result
        if (request.getParameter("itemResourceType") == null) {
            return hits.toString();
        }

        StringResponseWrapper hitResponse = new StringResponseWrapper(response);

        RequestDispatcherOptions requestDispatcherOptions = new RequestDispatcherOptions(null);

        // Force a resource type, to render the resource in a specific way
        requestDispatcherOptions.setForceResourceType(request.getParameter("itemResourceType"));

        for (Hit hit : hits) {
            String path = (String) hit.get("path");
            if (path != null) {
                Resource resource = request.getResourceResolver().getResource(path);
                if (resource != null) {
                    request.setAttribute(Resource.class.getCanonicalName(), resource);
                    RequestDispatcher dispatcher = request.getRequestDispatcher(resource.getPath(),
                            requestDispatcherOptions);
                    dispatcher.include(request, hitResponse);
                    request.removeAttribute(Resource.class.getCanonicalName());
                }
            }
        }
        return hitResponse.getString();
    }

    /**
     * Generates a GQL search expression to search tagged resources with tag title
     * @param title The tag title to be searched
     * @param tagPropertyPath Tag property on the resource
     * @param tagMgr TagManager
     * @param locale to search in a certain localized tag title only; use <code>null</code> to search in the default title
     * @return GQL Search Expression for the tag title search
     */
    protected StringBuilder getTagTitleGQLSearchExpression(final String title, final String tagPropertyPath,
        final TagManager tagMgr, final Locale locale) {

        StringBuilder titleQuery = new StringBuilder();
        Tag[] tags = tagMgr.findTagsByTitle(title, locale);

        int flag = 0;
        for(Tag tag : tags) {
            if(flag == 1) {
                titleQuery.append("OR ");
            } else {
                flag = 1;
            }
            titleQuery.append(tag.getGQLSearchExpression(tagPropertyPath));
            titleQuery.append(" ");
        }
        return titleQuery;
    }

    /**
     * As GQL supports only wildcards in the name property, we have to make some substitutions
     * so these wildcards aren't stripped out by gql
     * @param queryString
     * @return
     */
    protected static String preserveWildcards(String queryString) {
        // Replace "*" by "%", * so
        if (queryString != null) {
            queryString = queryString.replace("*", "%");
        }
        return queryString;
    }

    /**
     * As GQL supports only wildcards in the name property, we have to revert our substitutions
     * @param queryString
     * @return
     */
    protected static String revertWildcards(String queryString) {
        if (queryString != null) {
            // Replace "%" back to "*", to start a wildcard search
            queryString = queryString.replace("%", "*");
        }
        return queryString;
    }
}
