/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.core.stats;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import javax.jcr.Item;
import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.jackrabbit.util.Text;

import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.api.WCMMode;
import com.day.crx.statistics.Report;

/**
 * <code>PageViewReport</code> implements a report that returns historical data
 * about {@link Page} views.
 * 
 * @author asaar
 */
public class PageViewReport extends Report {

    /** Default period is 30 days*/
    private int period = 30;

    /** The page */
    private final Page page;
    
    /** The WCM mode */
    private final WCMMode mode;

    /**
     * Creates a new report.
     * 
     * @param dataPath
     *            the location where the result statistics are stored.
     * @param page
     *            the page that was viewed
     * @param mode
     *            the WCM mode that was used to view the page
     */
    public PageViewReport(String dataPath, Page page, WCMMode mode) {
        super(dataPath);
        this.page = page;
        this.mode = mode;
    }

    /**
     * {@inheritDoc} Returns result rows with the following objects:
     * <ul>
     * <li>Path <code>String</code>, which indicates the day</li>
     * <li><code>Long</code> count :how many times the page has been viewed on thath day</li>
     * <li><code>Long</code> rollingCount: how many times the page has been viewed in the last 30days</li>
     * </ul>
     */
    public Iterator<Object[]> getResult(Session session)
            throws RepositoryException {

        PageView view = new PageView(getDataPath(), page, mode);

        //skip if no statistics at all
        if(!session.itemExists(Text.getRelativeParent(view.getPath(),4))) {
            return Collections.<Object[]>emptySet().iterator();
        }

        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.DAY_OF_MONTH, -(getPeriod()-1));
        List<Object[]> data = new ArrayList<Object[]>();
        for (int i = 0; i < getPeriod(); i++) {
            view.setTimestamp(cal.getTimeInMillis());
            long count = 0, rolling = 0;
            Node node = getNode(view, session);
            if (node!=null) {
                count = node.getProperty(PageView.VIEWS).getLong();
                rolling = node.getProperty(PageView.ROLLING_MONTH_COUNT).getLong();
            } else {
                //if there is no rolling, this may be due to missing hit on that day.
                //search history => rolling-month check last 30 days
                Calendar back = Calendar.getInstance();
                back.setTimeInMillis(cal.getTimeInMillis());
                int j=1;
                for(;j<30;j++) {
                    back.add(Calendar.DAY_OF_MONTH,-1);
                    view.setTimestamp(back.getTimeInMillis());
                    node = getNode(view, session);
                    if (node!=null && node.hasProperty(PageView.ROLLING_MONTH_COUNT)) {
                        rolling = node.getProperty(PageView.ROLLING_MONTH_COUNT).getLong();
                        break;
                    }
                }
                //now substact the amount of hits contained in this rolling that are not part of the current
                if (rolling>0){
                    back.add(Calendar.DAY_OF_MONTH,-29);
                    while(j>0) {
                        view.setTimestamp(back.getTimeInMillis());
                        node = getNode(view, session);
                        if (node!=null && node.hasProperty(PageView.VIEWS)) {
                             rolling -= node.getProperty(PageView.VIEWS).getLong();
                        }
                        back.add(Calendar.DAY_OF_MONTH, 1);
                        j--;
                    }
                }
                view.setTimestamp(cal.getTimeInMillis());
            }
            data.add(new Object[] { view.getPath(), count, rolling});
            cal.add(Calendar.DAY_OF_MONTH, 1);
        }
        return data.iterator();
    }

    /** @return the report period in number of days */
    public int getPeriod() {
        return period;
    }

    /** @param period the report period in number of days */
    public void setPeriod(int period) {
        this.period = period;
    }

    protected Page getPage() {
        return page;
    }

    protected WCMMode getMode() {
        return mode;
    }

    private Node getNode(PageView view, Session session) {
        try {
            if (session.itemExists(view.getPath())) {
                Item item = session.getItem(view.getPath());
                if (item.isNode()) {
                    return (Node) item;
                }
            }
        } catch (RepositoryException e) {
            //todo:log
        }
        return null;
    }
}
