/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.designimporter;

import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.designimporter.api.CanvasBuilder;
import com.day.cq.wcm.designimporter.util.ComponentSuffixGenerator;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.Resource;
import org.osgi.framework.BundleContext;

import javax.jcr.Node;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * The context that encapsulates the information related to a design import request.
 *
 * <p>
 * A context object is created for an import request and passed along to various classes
 * participating in the import process. The participant classes (e.g. TagHandlers) can use
 * the contextual information as per their needs.
 * </p>
 */
public class DesignImporterContext {
    /**
     * Represents the design node that gets created during the design import process.
     *
     * <p>
     * While importing a design package, the design importer unloads all the referenced
     * assets(images, icons, styles, scripts etc.) into a design node that is uniquely
     * associated with the page under construction. This node acts as the design path
     * for the page that gets constructed after the import process.
     * </p>
     */
    public Node designNode;

    /**
     * Represents the CQ Page that initiates the import request
     *
     * <p>
     * The current import workflow involves dropping of the design package on an existing
     * page with a drop zone. The existing page itself gets transformed into the resultant
     * page.
     * </p>
     */
    public Page page;

    /**
     * The page being worked upon. This could be different from {@link #page} for secondary canvases
     */
    public Page currentPage;

    /**
     * The suffix generator that generates suffixes per component names to uniquely identify
     * them in the node hierarchy.
     */
    public ComponentSuffixGenerator componentSuffixGenerator;

    /**
     * The CanvasBuilder object associated with the current import request
     */
    public CanvasBuilder canvasBuilder;

    /**
     * The name of the HTML file for which the CQ page is getting constructed.
     *
     * <p>
     * The design package should typically contain an index.html file at archive root level.
     * However, it is possible to create multiple CQ pages from one design package itself.
     * The mobile page creation exemplifies this fact. The htmlName identifies the html document
     * the CQ page for which is getting constructed.
     * </p>
     */
    public String htmlName;

    /**
     * The OSGi BundleContext that may be used by the TagHandlers to interact with the OSGi framework
     */
    public BundleContext bundleContext;

    /**
     * The sling request object associated with the current import request.
     *
     * @deprecated Usage of this field has been removed (keeping this field for backward compatibility)
     */
    @Deprecated
    public SlingHttpServletRequest slingHttpServletRequest;

    /**
     * The importer resource associated with the current import operation
     */
    private Resource importer;


    private ArrayList<String> extractedResources;

    public static final String CHAR_ENCODING_KEY = "_character_encoding"; 

    /**
     * @param page
     * @param designNode
     */
    public DesignImporterContext(Page page, Node designNode) {
        this(page, designNode, null);
    }

    /**
     * Constructor
     *
     * @param page
     * @param designNode
     * @param htmlName
     */
    public DesignImporterContext(Page page, Node designNode, String htmlName) {
        this(page, designNode, htmlName, null);
    }

    public DesignImporterContext(Page page, Node designNode, String htmlName, CanvasBuilder canvasBuilder) {
        this(page, designNode, htmlName, canvasBuilder, null, new ArrayList<String>());
    }

    public DesignImporterContext(Page page, Node designNode, String htmlName, CanvasBuilder canvasBuilder, BundleContext bundleContext) {
        this(page, designNode, htmlName, canvasBuilder, bundleContext, new ArrayList<String>());
    }

    public DesignImporterContext(Page page, Node designNode, String htmlName, CanvasBuilder canvasBuilder, BundleContext bundleContext, ArrayList<String> extractedResources) {
        this.page = this.currentPage = page;
        this.designNode = designNode;
        this.htmlName = htmlName;
        this.canvasBuilder = canvasBuilder;
        this.bundleContext = bundleContext;
        this.extractedResources = extractedResources;

        componentSuffixGenerator = new ComponentSuffixGenerator();
    }

    /**
     * Placeholder for import warnings. Various participating classes may choose to log import errors
     * in this list.
     */
    public List<String> importWarnings = new ArrayList<String>();

    private Map<Object, Object> attributes = new HashMap<Object, Object>();

    public Map<Object, Object> getAttributes() {
        return attributes;
    }

    public Object getAttribute(Object key) {
        return attributes.get(key);
    }

    public Object setAttribute(Object key, Object value) {
        return attributes.put(key, value);
    }

    public void setImporter(Resource importer) {
        this.importer = importer;
    }

    public Resource getImporter() {
        return  importer;
    }

    public ArrayList<String> getExtractedResources() { return extractedResources; }

}
