/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.designimporter.parser.taghandlers;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import com.day.cq.commons.jcr.JcrConstants;
import org.apache.jackrabbit.commons.JcrUtils;

import com.day.cq.commons.jcr.JcrUtil;
import com.day.cq.wcm.designimporter.parser.HTMLContent;
import com.day.cq.wcm.designimporter.parser.HTMLContentType;

public class ClientLibsBuilderImpl {

    static private final String MIMETYPE_CSS = "text/css";

    static private final String MIMETYPE_JS = "application/javascript";

    private static final String MIMETYPE_TEXT_PLAIN = "text/plain";

    private Node designRoot;

    private String folderName;

    private String categoryName;

    private HTMLContent htmlContent;

    private String inlineScriptsFileName;

    private String inlineStylesFileName;

    private ArrayList<String> potentialReferences;

    public void setDesignRoot(Node designRoot) {
        this.designRoot = designRoot;
    }

    public void setFolderName(String folder) {
        this.folderName = folder;
    }

    public void setCategoryName(String categoryName) {
        this.categoryName = categoryName;
    }

    public void setHtmlContent(HTMLContent htmlContent) {
        this.htmlContent = htmlContent;
    }

    public void setInlineScriptsFileName(String fileName) {
        this.inlineScriptsFileName = fileName;
    }

    public void setInlineStylesFileName(String fileName) {
        this.inlineStylesFileName = fileName;
    }

    public void setPotentialReferences(ArrayList<String> potentialReferences) {
        this.potentialReferences = potentialReferences;
    }

    @SuppressWarnings("unchecked")
    public void build() throws RepositoryException {
        if (htmlContent != null) {

            // create clientlib node
            Node clientLib = JcrUtils.getOrAddNode(designRoot, folderName, "cq:ClientLibraryFolder");

            // extract js file names
            List<String> referencedScripts = (List<String>) htmlContent.get(HTMLContentType.SCRIPT_INCLUDE);
            ArrayList<String> extractedScriptNames = new ArrayList<String>(referencedScripts);

            // move js files into the clientlib
            copyFiles(designRoot, extractedScriptNames, clientLib, true, false);

            // extract css file names
            List<String> referencedStyleSheets = (List<String>) htmlContent.get(HTMLContentType.STYLESHEET_INCLUDE);
            ArrayList<String> extractedCssNames = new ArrayList<String>(referencedStyleSheets);

            // move css files into the clientlib
            copyFiles(designRoot, extractedCssNames, clientLib, true, true);

            // Add the inline scripts and styles to pageScripts.js and pageStyles.css respectively.
            String cssContent = (String) htmlContent.get(HTMLContentType.STYLES_INLINE);
            cssContent = doReferenceAdjustment(cssContent);
            String scriptContent = (String) htmlContent.get(HTMLContentType.SCRIPT_INLINE);

            JcrUtils.putFile(clientLib, inlineStylesFileName, MIMETYPE_CSS, new ByteArrayInputStream(cssContent.getBytes()));
            JcrUtils.putFile(clientLib, inlineScriptsFileName, MIMETYPE_JS, new ByteArrayInputStream(scriptContent.getBytes()));

            // finalize the clientlib

            // add inline script and css filename to the list of files
            extractedScriptNames.add(inlineScriptsFileName);
            extractedCssNames.add(inlineStylesFileName);

            String jsTxtFileContent = convertToLineSeparatedNames(extractedScriptNames);
            String cssTxtFileContent = convertToLineSeparatedNames(extractedCssNames);

            // write down category, js.txt and css.txt
            JcrUtil.setProperty(clientLib, "categories", new String[] { categoryName });
            JcrUtils.putFile(clientLib, "js.txt", MIMETYPE_TEXT_PLAIN, new ByteArrayInputStream(jsTxtFileContent.getBytes()));
            JcrUtils.putFile(clientLib, "css.txt", MIMETYPE_TEXT_PLAIN, new ByteArrayInputStream(cssTxtFileContent.getBytes()));

        }
    }

    private String convertToLineSeparatedNames(List<String> fileNames) {
        String content = "";
        for (String file : fileNames) {
            content += file + "\n";
        }
        return content;
    }

    private void copyFiles(Node parentSource, ArrayList<String> fileNames, Node destination, boolean deleteSource, boolean adjust) throws RepositoryException {
        for (String file : fileNames) {
            if ( parentSource.hasNode(file)) {
                Node source = parentSource.getNode(file);
                if (source != null) {
                    // create intermediate folders
                    ArrayList<String> folders = new ArrayList<String>();
                    Node parent = source.getParent();
                    while (!parent.getPath().equals(parentSource.getPath())) {
                        // found a folder
                        folders.add(parent.getName());
                        parent = parent.getParent();
                    }

                    // recreate folder structure in destination
                    Node finalDestination = destination;
                    for (int i = folders.size() - 1; i >= 0; i--) {
                        finalDestination = JcrUtils.getOrAddFolder(finalDestination, folders.get(i));
                    }

                    // copy file
                    Node copy = JcrUtil.copy(source, finalDestination, source.getName());

                    if ( adjust ) {
                        // since we moved the file, we might need to update the potential references
                        doReferenceAdjustment(copy);
                    }

                    if (deleteSource) {
                        parent = source.getParent();
                        source.remove();

                        // remove empty parent folders to cleanup
                        while ( !parent.getNodes().hasNext() ) {
                            Node p = parent;
                            parent = parent.getParent();
                            p.remove();
                        }

                        parent.getParent().getSession().save();
                    }
                }
            }
        }
    }

    private void doReferenceAdjustment(Node node) throws RepositoryException {
        if (node == null || !node.hasNode(JcrConstants.JCR_CONTENT) ||
                !node.hasProperty(JcrConstants.JCR_CONTENT + "/" + JcrConstants.JCR_DATA)) {
            return;
        }
        Node jcrContent = node.getNode(JcrConstants.JCR_CONTENT);
        String content = jcrContent.getProperty(JcrConstants.JCR_DATA).getString();
        content = doReferenceAdjustment(content);
        jcrContent.setProperty(JcrConstants.JCR_DATA, content);
        node.getSession().save();
    }

    // appends ../ to all references
    private String doReferenceAdjustment(String content) {
        String result = content;
        if ( this.potentialReferences != null && result != null) {
            for(String resource: potentialReferences) {
                result = result.replaceAll(resource, "../" + resource);
            }
        }
        return result;
    }

}
