package com.day.cq.wcm.msm.commons;

import java.util.Collections;
import java.util.Set;
import java.util.regex.Pattern;

import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.nodetype.NodeType;

import com.day.cq.wcm.msm.api.RolloutManager;

/**
 * Provides item filter capabilities through a set of patters that allow
 * excluding of nodes, properties and node types based on a set of patterns.
 */
public class ItemFilterImpl {

    private final Set<Pattern> nodeTypePattern;
    private final Set<Pattern> nodePattern;
    private final Set<Pattern> propPattern;
    private final RolloutManager rolloutManager;
    private final ItemFilterImpl defaultFilter;

    private ItemFilterImpl(RolloutManager rolloutManager) {
        this(null, null, null, rolloutManager, null);
    }

    ItemFilterImpl(Set<Pattern> nodeTypePattern,
                   Set<Pattern> nodePattern,
                   Set<Pattern> propPattern,
                   RolloutManager rolloutManager) {

        this(nodeTypePattern, nodePattern, propPattern, rolloutManager, new RolloutManagerFilter(rolloutManager));
    }

    ItemFilterImpl(Set<Pattern> nodeTypePattern,
                   Set<Pattern> nodePattern,
                   Set<Pattern> propPattern,
                   RolloutManager rolloutManager,
                   ItemFilterImpl defaultFilter) {

        this.nodeTypePattern = nodeTypePattern;
        this.nodePattern = nodePattern;
        this.propPattern = propPattern;
        this.rolloutManager = rolloutManager;
        this.defaultFilter = defaultFilter;
    }

    /**
     * Checks if a given property is excluded.
     *
     * @param property the {@code Property} to be checked
     * @return True in the following cases:
     *         - property is protected or automatically created
     *         - property pattern is empty and the default filter's excludes returns true
     *         - property's parent node is excluded OR
     *         the property name is a MSM reserved property (as indicated by the RolloutManager) OR
     *         one of the property patterns matches the property's name
     * @throws javax.jcr.RepositoryException
     */
    public boolean excludes(Property property) throws RepositoryException {
        if(property.getDefinition().isProtected() || property.getDefinition().isAutoCreated()) {
            return true;
        }
        if(propPattern.isEmpty()) {
            return defaultFilter.excludes(property);
        } else {
            String name = property.getName();
            return excludes(property.getParent())
                    || rolloutManager.isReservedProperty(name)
                    || matchName(name, propPattern);
        }
    }

    /**
     * Checks if a given node is excluded
     *
     * @param node the {@code Node} to be checked
     * @return True in the following cases:
     *         - node is protected
     *         - node name is a MSM reserved property
     *         - node type or mixin type is excluded
     *         - name matches one of the patterns in the node pattern set of this filter
     *         or the default filter if this filter is empty
     * @throws javax.jcr.RepositoryException
     */
    public boolean excludes(Node node) throws RepositoryException {
        String name = node.getName();
        boolean excludes = node.getDefinition().isProtected()
                || rolloutManager.isReservedProperty(name)
                || excludesNodeType(node.getPrimaryNodeType())
                || excludesNodeType(node.getMixinNodeTypes());
        if(!excludes) {
            Set<Pattern> pattern = (nodePattern==null) ? defaultFilter.getNodeNamePattern() : nodePattern;
            excludes = matchName(name, pattern);
        }
        return excludes;
    }

    /**
     * Checks if a given node type name is excluded
     *
     * @param nodeTypeName a {@code String} containing the node type
     * @return True if:
     *         - default filter excludeNodeType returns true, when this node type pattern set is empty
     *         - node type name matches one of patterns in the node type pattern set
     */
    public boolean excludesNodeType(String nodeTypeName) {
        if(nodeTypePattern.isEmpty()) {
            return defaultFilter.excludesNodeType(nodeTypeName);
        } else {
            return matchName(nodeTypeName, nodeTypePattern);
        }
    }

    public Set<Pattern> getNodeTypePattern() {
        return Collections.unmodifiableSet(nodeTypePattern);
    }

    public Set<Pattern> getNodeNamePattern() {
        return Collections.unmodifiableSet(nodePattern);
    }

    public Set<Pattern> getPropertyNamePattern() {
        return Collections.unmodifiableSet(propPattern);
    }

    private boolean excludesNodeType(NodeType... nodeTypes) {
        boolean excludes = false;
        for(int i = 0; !excludes && i<nodeTypes.length; i++) {
            NodeType nodeType = nodeTypes[i];
            excludes = excludesNodeType(nodeType.getName())
                    || excludesNodeType(nodeType.getSupertypes());
        }
        return excludes;
    }

    private boolean matchName(String name, Set<Pattern> patterns) {
        if(patterns!=null) {
            for(Pattern pattern : patterns) {
                if(pattern.matcher(name).matches()) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Rollout's manager item filter
     * Will delegate the excludes method calls to the {@code RolloutManager} service
     */
    private static final class RolloutManagerFilter extends ItemFilterImpl {

        private final RolloutManager rolloutManager;

        private RolloutManagerFilter(RolloutManager rolloutManager) {
            super(rolloutManager);
            this.rolloutManager = rolloutManager;
        }

        @Override
        public boolean excludes(Property property) throws RepositoryException {
            return rolloutManager.isReservedProperty(property.getName())
                    || rolloutManager.isExcludedProperty(BaseAction.isPage(property.getParent()), property.getName());
        }

        @Override
        public boolean excludes(Node node) throws RepositoryException {
            return rolloutManager.isExcludedNode(node)
                    || (!BaseAction.isPage(node) && rolloutManager.isExcludedParagraphProperty(node.getName()));
        }

        @Override
        public boolean excludesNodeType(String nodeTypeName) {
            return rolloutManager.isExcludedNodeType(nodeTypeName);
        }
    }
}
