package com.day.cq.wcm.msm.commons;

import java.util.Collections;
import java.util.Dictionary;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.commons.osgi.PropertiesUtil;

import com.day.cq.wcm.msm.api.RolloutManager;

public class ItemFilterUtil {

    public static final String[] EMPTY_PROPERTIES = new String[0];

    /**
     * Excluded properties parameter name to use for your Service Configuration.
     */
    public static final String PARAMETER_EXCLUDED_PROPERTIES = "cq.wcm.msm.action.excludedprops";

    public static final String[] EXCLUDED_PROPERTIES_DEFAULT = {"jcr:.*", "sling:.*", "cq:.*"};

    /**
     * Excluded paragraph items parameter name to use for your Service Configuration.
     */
    public static final String PARAMETER_EXCLUDED_PARAGRAPH_ITEMS = "cq.wcm.msm.action.excludedparagraphitems";

    public static final String[] EXCLUDED_PARAGRAPH_ITEMS_DEFAULT = {"cq:propertyInheritanceCancelled"};

    /**
     * Excluded node types parameter name to use for your Service Configuration.
     */
    public static final String PARAMETER_EXCLUDED_NODE_TYPES = "cq.wcm.msm.action.excludednodetypes";

    public static final String[] EXCLUDED_NODE_TYPES_DEFAULT = {
            "cq:LiveSyncAction", "cq:LiveSyncConfig", "cq:BlueprintSyncConfig"};


    /**
     * Creates an ItemFilterImpl from the given Properties with the name {@link #PARAMETER_EXCLUDED_NODE_TYPES} and
     * {@link #PARAMETER_EXCLUDED_PROPERTIES} <br/>
     * Properties default to empty.
     *
     * @param properties     used for configuration
     * @param rolloutManager to access the roll-out reserved settings.
     * @return ItemFilterImpl.
     */
    public static ItemFilterImpl createPageFilter(Dictionary properties, RolloutManager rolloutManager) {
        String[] pageProperties = PropertiesUtil.toStringArray(properties.get(PARAMETER_EXCLUDED_PROPERTIES),
                EMPTY_PROPERTIES);

        String[] nodeTypes = PropertiesUtil.toStringArray(properties.get(PARAMETER_EXCLUDED_NODE_TYPES),
                EMPTY_PROPERTIES);
        return createFilter(nodeTypes, null, pageProperties, rolloutManager);
    }

    /**
     * Creates an ItemFilterImpl from the given Properties with the name {@link #PARAMETER_EXCLUDED_NODE_TYPES} and
     * {@link #PARAMETER_EXCLUDED_PARAGRAPH_ITEMS} <br/>
     * Properties default to empty.
     *
     * @param properties     used for configuration
     * @param rolloutManager to access the roll-out reserved settings.
     * @return ItemFilterImpl.
     */
    public static ItemFilterImpl createComponentFilter(Dictionary properties, RolloutManager rolloutManager) {
        String[] paragraphItems = PropertiesUtil.toStringArray(properties.get(PARAMETER_EXCLUDED_PARAGRAPH_ITEMS),
                EMPTY_PROPERTIES);

        String[] nodeTypes = PropertiesUtil.toStringArray(properties.get(PARAMETER_EXCLUDED_NODE_TYPES),
                EMPTY_PROPERTIES);
        return createFilter(nodeTypes, paragraphItems, paragraphItems, rolloutManager);
    }

    /**
     * Helper method to create a page filter using a {@code ValueMap}
     *
     * @see #createPageFilter(java.util.Dictionary, com.day.cq.wcm.msm.api.RolloutManager)
     */
    public static ItemFilterImpl createPageFilter(ValueMap config, RolloutManager rolloutManager) {
        return createPageFilter(convertToDictionary(config), rolloutManager);
    }

    /**
     * Helper method to create a component filter using a {@code ValueMap}
     *
     * @see #createComponentFilter(java.util.Dictionary, com.day.cq.wcm.msm.api.RolloutManager)
     */
    public static ItemFilterImpl createComponentFilter(ValueMap config, RolloutManager rolloutManager) {
        return createComponentFilter(convertToDictionary(config), rolloutManager);
    }

    private static Dictionary convertToDictionary(Map<String, Object> map) {
        Dictionary dictionary = new Hashtable();
        for (Map.Entry<String, Object> entry : map.entrySet()) {
            dictionary.put(entry.getKey(), entry.getValue());
        }
        return dictionary;
    }

    /**
     * Creates an ItemFilterImpl based on a set of patterns that will be used to match NodeTypes, Node names and
     * Property names.
     *
     * @param nodeTypes      RegExp pattern of NodeType names to filter
     * @param nodeNames      RegExp pattern of Node names to filter
     * @param propertyNames  RegExp pattern of Property Names to filter
     * @param rolloutManager a {@code RolloutManager} to serve as the default filter if the above pattern sets are empty
     * @return an {@code ItemFilterImpl} instance
     */
    public static ItemFilterImpl createFilter(String[] nodeTypes,
                                              String[] nodeNames,
                                              String[] propertyNames,
                                              RolloutManager rolloutManager) {

        return new ItemFilterImpl(buildPatterns(nodeTypes),
                buildPatterns(nodeNames),
                buildPatterns(propertyNames),
                rolloutManager);
    }

    private static Set<Pattern> buildPatterns(String... patterns) {
        if(patterns!=null && patterns.length>0) {
            final Set<Pattern> pats = new HashSet<Pattern>(patterns.length);
            for(String pattern : patterns) {
                pats.add(Pattern.compile(pattern));
            }
            return Collections.unmodifiableSet(pats);
        } else {
            return Collections.emptySet();
        }
    }
}