/*
 * Copyright 1997-2010 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.notification;

import org.apache.felix.scr.annotations.Component;
import org.apache.sling.api.resource.PersistableValueMap;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

@Component(componentAbstract = true)
public abstract class AbstractSubscription implements Subscription {


    public static final class Filter {

        /**
         * The actions.
         */
        public final String[] actions;

        /**
         * The path entries.
         */
        public final Entry[] list;

        /**
         * The configuration.
         */
        final PersistableValueMap configuration;

        /**
         * Creates a new filter with the given filter list and startpoints
         *
         * @param config The notification subscription config
         */
        public Filter(final PersistableValueMap config) {
            // actions
            this.actions = config.get("actions", new String[]{});
            // path
            final List<Entry> entries = new ArrayList<Entry>();
            final String[] paths = config.get("paths", new String[]{});
            for (final String p : paths) {
                int p1 = p.indexOf('|');
                int p2 = p.lastIndexOf('|');
                final boolean exact = Boolean.valueOf(p.substring(p1 + 1, p2));
                final boolean allow = Boolean.valueOf(p.substring(p2 + 1));
                final Entry entry = new Entry(p.substring(0, p1), exact, allow);
                entries.add(entry);
            }
            this.list = entries.toArray(new Entry[entries.size()]);
            this.configuration = config;
        }

        public PersistableValueMap getConfiguration() {
            return this.configuration;
        }

        public String[] getActions() {
            return this.actions;
        }

        /**
         * Two <code>Filter</code> objects are considered equal if they contain equal {@link Entry}s in the same order.
         * Furthermore they are only considered equals if they are both compiled OR both are not compiled!
         *
         * @param obj the other <code>Filter</code>.
         *
         * @return <code>true</code> if <code>obj</code> is equal to <code>this</code> <code>Filter</code>.
         */
        public boolean equals(Object obj) {
            return this == obj || obj instanceof Filter && Arrays.equals(this.list, ((Filter) obj).list);
        }

        /**
         * Returns a hashCode for this <code>Filter</code>.<br> Please keep in mind that this hashCode may change when
         * this <code>Filter</code> is modified!
         *
         * @return Returns a hashCode for this <code>Filter</code>.
         */
        public int hashCode() {
            return Arrays.asList(list).hashCode();
        }

        public Entry[] getEntries() {
            return this.list;
        }
    }

    /**
     * The <code>Entry</code> innerclass is needed to hold the filter and the <emp>sign</emp> of the rule.
     */
    public static final class Entry {

        /**
         * The path.
         */
        public final String path;

        /**
         * allow or deny
         */
        public final boolean allow;

        /**
         * exact or tree
         */
        public final boolean exact;

        /**
         * Constructs a new Entry
         *
         * @param path  The path to match.
         * @param exact Whether an exact match is desired.
         * @param allow Whether this is an allow- or deny-rule.
         */
        public Entry(String path, boolean exact, boolean allow) {
            this.path = path;
            this.exact = exact;
            this.allow = allow;
        }

        /**
         * Returns the allow flag
         *
         * @return <code>true</code> if this entry defines a positive rule; <code>false</code> otherwise.
         */
        public boolean isAllow() {
            return allow;
        }

        /**
         * Checks, if the path satisfies the rule of this entry.
         *
         * @param path the path to check
         * @param def  the value to return, if the handle does not satisfy the rule.
         *
         * @return <code>true</code> if the filter matches the handle; the value of <code>def</code> otherwise.
         */
        public boolean apply(String path, boolean def) {
            final boolean match;
            if (this.exact) {
                match = this.path.equals(path);
            } else {
                match = this.path.equals(path) || path.startsWith((this.path.equals("/") ? "" : this.path + "/"));
            }
            return match ? allow : def;
        }

        /**
         * Returns a human readable form of this entry
         */
        public String toString() {
            return (allow ? "allow" : "deny") + " " + path + " " + (exact ? "exact" : "tree");
        }

    }
}
