/*
 * Copyright 2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.offline;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;

import org.apache.poi.hwpf.HWPFDocument;
import org.apache.poi.hwpf.model.StyleDescription;
import org.apache.poi.hwpf.model.StyleSheet;
import org.apache.poi.hwpf.usermodel.Paragraph;
import org.apache.poi.hwpf.usermodel.Range;

/**
 * Exporter class for converting CQ5 WCM content to MS Word documents.
 * <p/>
 * Note: It seems that it is not possible to export pictures using poi
 */
public class OfflineExporter {

    private class Style {

        private int index;

        private StyleDescription description;

        public Style(int index, StyleDescription description) {
            this.index = index;
            this.description = description;
        }

        public Paragraph addParagraph() {
            return range.insertAfter(description.getPAP(), index);
        }

    }

    public static void export(Node node, int maxLevel, OutputStream stream)
            throws IOException, RepositoryException {
        OfflineExporter exporter = new OfflineExporter(maxLevel);

        if (node.hasProperty("jcr:content/jcr:title")) {
            exporter.document.getSummaryInformation().setTitle(
                    node.getProperty("jcr:content/jcr:title").toString());
        }

        exporter.exportPage(node, 0);

        exporter.document.write(stream);
    }

    private final HWPFDocument document;

    private final Range range;

    private final Map<String, Style> styles = new HashMap<String, Style>();

    private int maxLevel = 0;

    /**
     * Creates an exported instance. This instance can only be used for
     * exporting a single page (optionally with subpages), and should be
     * discarded once used.
     *
     * @param maxLevel maximum level of subpages to export
     * @throws IOException if the template Word document can not be loaded
     */
    private OfflineExporter(int maxLevel) throws IOException {
        InputStream stream =
            OfflineExporter.class.getResourceAsStream("export.doc");
        try {
            this.document = new HWPFDocument(stream);
            this.range = document.getRange();
            this.maxLevel = maxLevel;

            StyleSheet stylesheet = document.getStyleSheet();
            for (int i = 0; i < stylesheet.numStyles(); i++) {
                StyleDescription desc = stylesheet.getStyleDescription(i);
                if (desc != null) {
                    styles.put(desc.getName(), new Style(i, desc));
                }
            }
        } finally {
            stream.close();
        }
    }

    /**
     * Exports a CQ5 page. Subpages are exported if so configured.
     *
     * @param page cq:Page node
     * @param level current subpage level (0 is the top page)
     * @throws RepositoryException if a repository error occurs
     */
    private void exportPage(Node page, int level) throws RepositoryException {
        if (page.hasNode("jcr:content")) {
            exportContent(page.getNode("jcr:content"), level);
        }

        if (level < maxLevel) {
            NodeIterator iterator = page.getNodes();
            while (iterator.hasNext()) {
                Node child = iterator.nextNode();
                if (child.isNodeType("cq:Page")) {
                    exportPage(child, level + 1);
                }
            }
        }
    }

    /**
     * Exports the contents of the jcr:content node of a page.
     *
     * @param node jcr:content node
     * @param level current subpage level (0 is the top page)
     * @throws RepositoryException if a repository error occurs
     */
    private void exportContent(Node node, int level) throws RepositoryException {
        if (node.hasNode("par")) {
            NodeIterator iterator = node.getNode("par").getNodes();
            while (iterator.hasNext()) {
                Node child = iterator.nextNode();
                if (child.hasProperty("sling:resourceType")) {
                    String type =
                        child.getProperty("sling:resourceType").getString();
                    if (type.equals("foundation/components/title")) {
                        exportTitle(child, level);
                    } else if (type.equals("foundation/components/text")) {
                        exportText(child);
                    } else if (type.equals("foundation/components/text")) {
                        exportText(child);
                    } else if (type.equals("foundation/components/textimage")) {
                        exportText(child);
                    }
                }
            }
        }
    }

    /**
     * Exports a title paragraph.
     *
     * @param node title paragraph node
     * @param level current subpage level (0 is the top page)
     * @throws RepositoryException if a repository error occurs
     */
    private void exportTitle(Node node, int level) throws RepositoryException {
        if (node.hasProperty("type")
                && "small".equals(node.getProperty("type").getString())) {
            level++;
        }
        Style style = styles.get("Heading");

        if (style != null && node.hasProperty("jcr:title")) {
            Paragraph paragraph = style.addParagraph();
            paragraph.insertAfter(node.getProperty("jcr:title").getString());
        }
    }

    /**
     * Exports a text paragraph.
     *
     * @param node text paragraph node
     * @throws RepositoryException if a repository error occurs
     */
    private void exportText(Node node) throws RepositoryException {
        if (node.hasProperty("text") && styles.containsKey("Text body")) {
            Paragraph paragraph = styles.get("Text body").addParagraph();
            if (node.hasProperty("textIsRich")
                    && node.getProperty("textIsRich").getBoolean()) {
                paragraph.insertAfter(node.getProperty("text").getString());
            } else {
                paragraph.insertAfter(node.getProperty("text").getString());
            }
        }
    }

}
