/*
 * Copyright 2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.offline;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Calendar;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.sling.api.request.RequestParameter;
import org.apache.sling.api.resource.ValueMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.cq.commons.jcr.JcrConstants;
import com.day.cq.commons.jcr.JcrUtil;

/**
 * Importer class for converting MS Word documents to CQ5 pages.
 */
public class OfflineImporter {

    private static final Logger log = LoggerFactory.getLogger(OfflineImporter.class);

    private final Node parent;

    private final TextDocumentImporter importer;

    private String filename;

    private String resourceType;

    private String template;

    private ValueMap config;

    /**
     * Creates an importer for importing the given Word document to the given
     * target node.
     *
     * @param parent parent node under which the imported pages are created
     * @param stream stream containing the Word document
     * @throws IOException if the Word document could not be loaded
     * @throws TextImportException if the document uses an unknown format
     * @deprecated 
     */
    public OfflineImporter(Node parent, InputStream stream) throws IOException, TextImportException {
        this.parent = parent;
        this.importer = new DocImporter(stream);
    }

    /**
     * Creates an importer for importing the given Word document to the given
     * target node.
     *
     * @param parent parent node under which the imported pages are created
     * @param rp sling request parameter containing the document
     * @throws IOException if the Word document could not be loaded
     * @throws TextImportException if the document uses an unknown format
     */
    public OfflineImporter(Node parent, RequestParameter rp) throws IOException, TextImportException {
        this.parent = parent;
        this.importer = getImporter(rp);
    }

    private static TextDocumentImporter getImporter(RequestParameter rp) throws IOException, TextImportException {
        try {
            return new DocImporter(rp.getInputStream());
        }
        catch (TextImportException ex){
            return new DocxImporter(rp.getInputStream());
        }
    }

    public void setFilename(String filename) {
        this.filename = filename;
    }

    public void setResourceType(String resourceType) {
        this.resourceType = resourceType;
    }

    public void setTemplate(String template) {
        this.template = template;
    }

    /**
     * Imports the document to the given target node.
     *
     * @throws RepositoryException if a repository error occurs
     */
    public void importDocument() throws RepositoryException {
        String title = getTitle(filename);

        Node node = parent.addNode(getNodeName(filename), "cq:Page");

        Node content = node.addNode(JcrConstants.JCR_CONTENT, "cq:PageContent");
        if (resourceType != null) {
            content.setProperty("sling:resourceType", resourceType);
        }
        if (template != null) {
            content.setProperty("cq:template", template);
        }
        content.setProperty(JcrConstants.JCR_TITLE, title);
        content.setProperty("pageTitle", title);

        Node parsys = content.addNode("par", JcrConstants.NT_UNSTRUCTURED);
        parsys.setProperty("sling:resourceType", "foundation/components/parsys");

        for (int i = 0; i < this.importer.getNumberOfParagraphs(); i++) {
            Node paragraph = parsys.addNode("par" + i, "nt:unstructured");
            Paragraph p = this.importer.getParagraph(i);

            String html = p.getHTML();
            List<Picture> pics = p.getPictures(); 

            if (pics.size() == 0) {
                // no images
                importTextParagraph(paragraph, p);
            }
            else if (html.length() == 0) {
                // no text
                importImageParagraph(paragraph, p);
            }
            else {
                importTextImageParagraph(paragraph, p);
            }
        }
    }

    /**
     * Imports a paragraph to the given rich text paragraph node. 
     */
    private void importTextParagraph(Node node, Paragraph paragraph)
            throws RepositoryException {
        node.setProperty("sling:resourceType", config.get("text", "foundation/components/text"));
        node.setProperty("textIsRich", "true");
        node.setProperty("text", paragraph.getHTML());
    }

    /**
     * Imports a paragraph to the given rich text paragraph node.
     */
    private void importTextImageParagraph(Node node, Paragraph paragraph)
            throws RepositoryException {
        node.setProperty("sling:resourceType", config.get("textimage", "foundation/components/textimage"));
        node.setProperty("textIsRich", "true");
        Node image = node.addNode("image", JcrConstants.NT_UNSTRUCTURED);
        importImageParagraph(image, paragraph);
        node.setProperty("text", paragraph.getHTML());
    }

    /**
     * Imports a paragraph to the given rich text paragraph node.
     */
    private void importImageParagraph(Node node, Paragraph paragraph)
            throws RepositoryException {
        List<Picture> pics = paragraph.getPictures();
        if (pics.size() == 0) {
            throw new RepositoryException("Paragraoh does not contain any pictures");
        } else if (pics.size() > 1) {
            log.debug("paragraph contains multiple pictures, all but last are dropped");
        }
        node.setProperty("sling:resourceType", (String) config.get("image", "foundation/components/image"));
        Node file = node.addNode("file", JcrConstants.NT_RESOURCE);
        file.setProperty(JcrConstants.JCR_LASTMODIFIED, Calendar.getInstance());
        file.setProperty(JcrConstants.JCR_MIMETYPE, pics.get(0).getMediaType());
        file.setProperty(JcrConstants.JCR_DATA,
                node.getSession().getValueFactory().createBinary(new ByteArrayInputStream(pics.get(0).getBytes())));
    }

    private String getTitle(String filename) {
        String title = this.importer.getTitle();
        return title != null ? title : trimFilename(filename);
    }

    private String trimFilename(String filename) {
        if (filename.endsWith(".doc")) {
            return filename.substring(0, filename.length() - ".doc".length());
        } else if (filename.endsWith(".docx")) {
            return filename.substring(0, filename.length() - ".docx".length());
        } else {
            return filename;
        }
    }

    private String getNodeName(String filename) {
        return JcrUtil.createValidName(trimFilename(filename));
    }

    public void setConfig(ValueMap config) {
        this.config = config;
    }
}
