/*
 * Copyright 1997-2009 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.siteimporter;

import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.osgi.services.HttpClientBuilderFactory;
import org.apache.commons.lang.StringUtils;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.SlingHttpServletResponse;
import org.apache.sling.api.request.RequestParameter;
import org.apache.sling.api.resource.ResourceResolver;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.granite.xss.XSSAPI;

import java.io.IOException;
import java.io.PrintWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


public class ImporterContext implements ProgressTracker {
    private static final Logger log = LoggerFactory
            .getLogger(ImporterContext.class);

    static final String LANG_DEFAULT = "en";
    static final String USER_AGENT = "Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.7; en-US; rv:1.9.2.18) Gecko/20110614 Firefox/3.6.18";

    private SlingHttpServletRequest request;
    private SlingHttpServletResponse response;
    private ResourceResolver resolver;
    private ResourcesBoard board;
    private CloseableHttpClient httpClient;

    private String projectName;
    private String title;
    private URL original;
    private String sitemap;
    private String resourceSuperType;
    private boolean overwrite;
    private String defaultLanguage;

    private final List<ServiceReference> references = new ArrayList<ServiceReference>();
    private final Map<String, Object> services = new HashMap<String, Object>();

    private final List<String> paths = new ArrayList<String>();

    private BundleContext bundleContext;

    private ImporterContext() {
        // private constructor
    }

    public static ImporterContext createContext(
            SlingHttpServletRequest request, SlingHttpServletResponse response,
            BundleContext bundleContext) throws MalformedURLException {
        return ImporterContext.createContext(request, response, bundleContext,
                new ResourcesBoard());
    }

    public static ImporterContext createContext(
            SlingHttpServletRequest request, SlingHttpServletResponse response,
            BundleContext bundleContext, ResourcesBoard board)
            throws MalformedURLException {
    	
        // Allow circular redirects, because some cookie-check mechanisms do
        // a single "circle" and would cause an exception otherwise.
        // Limit MAX_REDIRECT instead to prevent endless redirect loops or traps.
    	
        RequestConfig rc = RequestConfig.custom().setCookieSpec(CookieSpecs.BROWSER_COMPATIBILITY)
		    .setCircularRedirectsAllowed(true)
		    .setMaxRedirects(5).build();
        
        ImporterContext ctx = new ImporterContext();
        ctx.bundleContext = bundleContext;
        
        ServiceReference sr = bundleContext.getServiceReference(HttpClientBuilderFactory.class.getName());
        HttpClientBuilderFactory httpClientBuilderFactory = null;
        if(sr != null) {
            httpClientBuilderFactory = (HttpClientBuilderFactory) bundleContext.getService(sr); 
        }
        
        if(httpClientBuilderFactory != null && httpClientBuilderFactory.newBuilder() != null) {
            ctx.httpClient = httpClientBuilderFactory.newBuilder()
                .setDefaultRequestConfig(rc)
                .setUserAgent(USER_AGENT)
                .build();
        } else {
            ctx.httpClient = HttpClients.custom().build();
        }
        
        ctx.request = request;
        ctx.response = response;
        ctx.resolver = request.getResourceResolver();
        ctx.board = board;

        ctx.projectName = request.getParameter("projectname").replace(" ", "")
                .toLowerCase();

        ctx.title = request.getParameter("projectname");
        ctx.sitemap = request.getParameter("sitemap");
        ctx.resourceSuperType = request.getParameter("resourceSuperType");

        ctx.overwrite = "true".equals(request.getParameter("overwrite"));
        ctx.original = new URL(request.getParameter("original"));

        return ctx;
    }

    public String getProjectName() {
        return projectName;
    }

    public String getTitle() {
        return title;
    }

    public URL getOriginal() {
        return original;
    }

    public String getSitemap() {
        return sitemap;
    }

    public String getResourceSuperType() {
        return resourceSuperType;
    }

    public boolean isOverwrite() {
        return overwrite;
    }

    public String getContextPath() {
        return request.getContextPath();
    }

    public ResourceResolver getResolver() {
        return resolver;
    }

    public ResourcesBoard getBoard() {
        return board;
    }

    public PrintWriter getWriter() {
        try {
            return response.getWriter();
        } catch (IOException e) {
            return null;
        }
    }

    public CloseableHttpClient getHttpClient() {
        return httpClient;
    }

    public List<String> getPaths() {
        return paths;
    }

    public void addPath(String path) {
        paths.add(path);
    }

    public RequestParameter getThumbnail() {
        return request.getRequestParameter("preview");
    }

    @SuppressWarnings("unchecked")
    public <T> T getService(Class<T> type) {
        T service = (T) this.services.get(type.getName());
        if (service == null) {
            final ServiceReference ref = this.bundleContext
                    .getServiceReference(type.getName());
            if (ref != null) {
                this.references.add(ref);
                service = (T) this.bundleContext.getService(ref);
                this.services.put(type.getName(), service);
            }
        }
        return service;
    }

    public void dispose() {
        for (final ServiceReference reference : this.references) {
            this.bundleContext.ungetService(reference);
        }
        this.references.clear();
        this.services.clear();
    }

    // ----< ProgressTracker interface >----------------------------------------

    private int currentIndent = 0;

    public void output(String message) {
        output(message, null);
    }

    public void output(String message, String css) {
        output(message, css, null);
    }

    private void output(String message, String css, Exception e) {
        if (e != null) {
            log.error(message, e);
        } else {
            log.info(message);
        }
        String script = "";
        if (ImporterStep.HEADING_CLASS.equals(css)) {
            script = " onclick=\"javascript:toggleDetails(this)\"";
        }
        css = ((css == null) ? "" : " class=\"" + css + "\"");
        try {
            ServiceReference sr = bundleContext.getServiceReference(XSSAPI.class.getName());
            XSSAPI xssapi = null;
            if(sr != null) {
                xssapi = (XSSAPI) bundleContext.getService(sr); 
            }
            response.getWriter().print(
                    "<li" + script + css + ">" + xssapi.encodeForHTML(message) + "</li>\n");

            response.getWriter().write("<script type=\"text/javascript\">");
            response.getWriter().write("window.scrollTo(0, 100000);");
            response.getWriter().write("</script>");

            response.flushBuffer();
        } catch (IOException ioe) {
            // TODO add logging
        }
    }

    public void error(String message) {
        output(message, "error", null);
    }

    public void error(String message, Exception e) {
        output(message, "error", e);
    }

    public void doIndent() {
        try {
            response.getWriter().print("<ol style=\"display:none\">\n");
            response.flushBuffer();
        } catch (IOException e) {
            // TODO add logging
        }
        currentIndent++;
    }

    public void undoIndent() {
        try {
            response.getWriter().print("</ol>\n");
            response.flushBuffer();
        } catch (IOException e) {
            // TODO add logging
        }
        currentIndent--;
    }

    public void repairIndent() {
        while (currentIndent > 0) {
            undoIndent();
        }
    }

    public void flushBuffer() {
        try {
            response.flushBuffer();
        } catch (IOException e) {
            // TODO add logging
        }
    }

    public void setDefaultLanguage(final String defaultLanguage) {
        this.defaultLanguage = defaultLanguage;
    }

    public String getDefaultLanguage() {
        return StringUtils.defaultIfEmpty(defaultLanguage, LANG_DEFAULT);
    }
}
