/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.workflow.process;

import com.day.cq.dam.api.Asset;
import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.api.PageManager;
import com.day.cq.wcm.api.Revision;
import com.day.cq.wcm.api.WCMException;
import com.day.cq.workflow.WorkflowException;
import com.day.cq.workflow.WorkflowSession;
import com.day.cq.workflow.exec.WorkItem;
import com.day.cq.workflow.exec.WorkflowData;
import com.day.cq.workflow.exec.WorkflowProcess;
import com.day.cq.workflow.job.AbsoluteTimeoutHandler;
import com.day.cq.workflow.metadata.MetaDataMap;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.Service;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.jcr.resource.JcrResourceResolverFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.version.Version;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

@Component(metatype = false)
@Service
@Property(name = "process.label", value = "Create Version")
public class CreateVersionProcess implements WorkflowProcess {

    /** the logger */
    private static final Logger log = LoggerFactory.getLogger(CreateVersionProcess.class);

    public static final String TYPE_JCR_PATH = "JCR_PATH";
    public static final String TYPE_JCR_UUID = "JCR_UUID";

    @Reference
    private JcrResourceResolverFactory jcrResolverFactory = null;

    public void execute(WorkItem workItem, WorkflowSession workflowSession, MetaDataMap args) throws WorkflowException {
        try {
            Session session = workflowSession.getSession();
            WorkflowData data = workItem.getWorkflowData();
            String path = null;
            String type = data.getPayloadType();
            if (type.equals(TYPE_JCR_PATH) && data.getPayload() != null) {
                String payloadData = (String) data.getPayload();
                if (session.itemExists(payloadData)) {
                    path = payloadData;
                }
            } else if (data.getPayload() != null && type.equals(TYPE_JCR_UUID)) {
                Node node = session.getNodeByIdentifier((String) data.getPayload());
                path = node.getPath();
            }

            if (path != null) {
                // create version for page or asset
                Resource res = jcrResolverFactory.getResourceResolver(session).getResource(path);
                String revStr = null;
                if (res.adaptTo(Page.class) != null) {
                    Page page = res.adaptTo(Page.class);
                    final PageManager pm = page.getPageManager();
                    final String versionLabel = createUniqueVersionLabel(pm.getRevisions(page.getPath(), null),
                                                                         getAbsTime(workItem));
                    Revision rev = pm.createRevision(page, versionLabel, null);
                    revStr = getPageRevision(rev, session);
                } else if (res.adaptTo(Asset.class) != null) {
                    Asset asset = res.adaptTo(Asset.class);
                    final String versionLabel = createUniqueVersionLabel(asset.getRevisions(null),
                                                                         getAbsTime(workItem));
                    com.day.cq.dam.api.Revision rev = asset.createRevision(versionLabel, null);
                    revStr = getAssetRevision(rev, session);
                } else {
                    log.warn("Cannot create version of " + path);
                }

                // set version info in workflow data's metadata
                if (revStr != null) {
                    data.getMetaDataMap().put("resourceVersion", revStr);
                    if (workItem.getWorkflowData().getMetaDataMap().get(AbsoluteTimeoutHandler.ABS_TIME, String.class) != null) {
                        Calendar cal = getTime(workItem);
                        data.getMetaDataMap().put("comment",
                                "Activate version " + revStr + " on " + cal.getTime().toString());
                    }
                    workflowSession.updateWorkflowData(workItem.getWorkflow(), data);
                }
            } else {
                log.warn("Cannot process payload. Path is null");
            }
        } catch (RepositoryException e) {
            throw new WorkflowException(e);
        } catch (WCMException e) {
            throw new WorkflowException(e);
        } catch (Exception e) {
            throw new WorkflowException(e);
        }
    }

    private String createUniqueVersionLabel(Collection<?> revisions, final String versionLabelHint)
            throws RepositoryException {

        if (versionLabelHint==null) {
            return null;
        }

        final List<Version> versions = new LinkedList<Version>();
        for (final Object o : revisions) {

            final Version v;
            if (o instanceof Revision) {
                v = ((Revision)o).getVersion();
            } else if (o instanceof com.day.cq.dam.api.Revision) {
                v = ((com.day.cq.dam.api.Revision)o).getVersion();
            } else {
                v = null;
            }

            if (null != v) {
                versions.add(v);
            }
        }

        String versionLabel = versionLabelHint;

        int count = 1;
        while (true) {

            boolean unique = true;
            for (final Version v : versions) {
                if (v.getContainingHistory().hasVersionLabel(versionLabel)) {
                    versionLabel = versionLabelHint + " (" + ++count + ")";
                    unique = false;
                    break;
                }
            }

            if (unique) {
                break;
            }
        }

        return versionLabel;
    }

    private String getPageRevision(Revision revision, Session session) throws RepositoryException {
        String vid = revision.getId();
        Node v = session.getNodeByIdentifier(vid);
        return v.getName();
    }

    private String getAssetRevision(com.day.cq.dam.api.Revision revision, Session session) throws RepositoryException {
        String vid = revision.getId();
        Node v = session.getNodeByIdentifier(vid);
        return v.getName();
    }

    private String getAbsTime(WorkItem workItem) {
        if (workItem.getWorkflowData().getMetaDataMap().get(AbsoluteTimeoutHandler.ABS_TIME, String.class) != null) {
            Calendar cal = getTime(workItem);
            SimpleDateFormat formatter = new SimpleDateFormat("yyyy.MM.dd HH.mm.ss");
            return "Scheduled Activation Time is " + formatter.format(cal.getTime());
        } else {
            return null;
        }
    }

    private Calendar getTime(WorkItem workItem) {
        Long time = workItem.getWorkflowData().getMetaDataMap().get(AbsoluteTimeoutHandler.ABS_TIME, Long.class);
        Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(time);
        return cal;
    }

}
