/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.workflow.collection;

import org.apache.jackrabbit.JcrConstants;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

/**
 * The <code>ResourceCollectionUtil</code> class provides some public utility
 * methods...
 */
public class ResourceCollectionUtil {

    /**
     * The method returns the {@link com.day.cq.workflow.collection.ResourceCollection}
     * which is either represented by the given node or one of its children.
     * This is useful in case the {@link com.day.cq.workflow.collection.ResourceCollection}
     * is somewhere placed on a cq page (e.g.: /aPage/jcr:content/cq:collection/...
     *
     * @param node possible resource collection node.
     * @param manager the {@link com.day.cq.workflow.collection.ResourceCollectionManager}
     * @return the {@link com.day.cq.workflow.collection.ResourceCollection} if
     * such a collection found otherwise <code>null</code>
     *
     * @throws RepositoryException
     */
    public static ResourceCollection getResourceCollection(Node node, ResourceCollectionManager manager)
            throws RepositoryException {
        // check if node represents a resource collection root node.
        if (!node.isNodeType("vlt:PackageDefinition")) {
            // this is not the case. search the child node entries.
            NodeIterator itr = node.getNodes();
            while (itr.hasNext()) {
                Node n = itr.nextNode();
                ResourceCollection coll = getResourceCollection(n, manager);
                if (coll != null) {
                    return coll;
                }
            }
        } else {
            return manager.createCollection(node);
        }
        return null;
    }

    /**
     * A {@link com.day.cq.workflow.collection.ResourceCollection} can be part of
     * a cq page. This method returns the containing cq page.
     *
     * @param collection the {@link com.day.cq.workflow.collection.ResourceCollection}
     * @param session the user {@link Session}
     * @return The page node if found otherwise the node which represents the collection.
     * @throws RepositoryException
     */
    public static Node getContainingPage(ResourceCollection collection, Session session)
            throws RepositoryException {
        if (collection.getPath().indexOf(JcrConstants.JCR_CONTENT) > 0) {
            Node node = (Node) session.getItem(collection.getPath());
            while (!node.getName().equals(JcrConstants.JCR_CONTENT)) {
                node = node.getParent();
            }
            return node.getParent();
        } else {
            // is this correct
            return (Node) session.getItem(collection.getPath());
        }
    }

}
