/*
 * Copyright 2010 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */

package com.day.cq.workflow.exec;


import com.day.cq.workflow.WorkflowException;
import com.day.cq.workflow.WorkflowSession;
import com.day.cq.workflow.metadata.MetaDataMap;

import java.io.Serializable;

/**
 * This is the interface to use to run external process steps.
 * <p/>
 * The external process steps are covered by {@link com.day.cq.workflow.model.WorkflowNode#TYPE_EXTERNAL_PROCESS}.
 * They are typically used for asynchronous tasks to execute on external systems.
 * <p/>
 * <b>Flow:</b></br>
 * <ol>
 * <li>First the {@link #execute(WorkItem, com.day.cq.workflow.WorkflowSession, com.day.cq.workflow.metadata.MetaDataMap)} method is called,
 * providing an identifier for the external process</li>
 * <li>Then the {@link #hasFinished(java.io.Serializable, WorkItem, com.day.cq.workflow.WorkflowSession, com.day.cq.workflow.metadata.MetaDataMap)}
 * method is called periodically to check if the external process has finished. Interval and timeouts can be configured on the step</li>
 * <li>Once {@link #hasFinished(java.io.Serializable, WorkItem, com.day.cq.workflow.WorkflowSession, com.day.cq.workflow.metadata.MetaDataMap)}
 * returns <code>true</code> the {@link #handleResult(java.io.Serializable, WorkItem, com.day.cq.workflow.WorkflowSession, com.day.cq.workflow.metadata.MetaDataMap)}
 * method is called, to read the results, if any</li>
 * </ol>
 * <b>Process Identifiers:</b></br>
 * It's up to the implementation how the external process identifier returned by <code>execute()</code> looks like. The engine does not make any assumptions about it, other than it
 * must be serializable. But of course the implementations of <code>hasFinished()</code> and <code>handleResult()</code> must know about it and handle it accordingly.
 *
 */
public interface WorkflowExternalProcess {

    /**
     * Executes the external process step. In order to keep track of this asynchronous process, an identifier must be returned. One can return <code>null</code>
     * to indicate that nothing was executed. In this case, the step directly advances to the next step.
     *
     * @param item      The current work item
     * @param session   The current workflow session
     * @param arguments The current steps node arguments (aka as node meta data)
     * @return Any serializable identifier for the external process executed, or <code>null</code>
     * @throws com.day.cq.workflow.WorkflowException
     */
    public Serializable execute(WorkItem item, WorkflowSession session, MetaDataMap arguments) throws WorkflowException;

    /**
     * Tests the external process, given by the external process id, for completeness.
     *
     * @param externalProcessId The external process to test for completeness
     * @param item              The current work item
     * @param session           The current workflow session
     * @param arguments         The current steps node arguments (aka as node meta data)
     * @return
     */
    public boolean hasFinished(Serializable externalProcessId, WorkItem item, WorkflowSession session, MetaDataMap arguments);

    /**
     * Get and handle the result of the external process, given by the external process id. This would include storing any results where appropriate for
     * further use.
     *
     * @param externalProcessId The external process to handle the results for
     * @param workItem          The current work item
     * @param session           The current workflow session
     * @param arguments         The current steps node arguments (aka as node meta data)
     * @throws com.day.cq.workflow.WorkflowException
     */
    public void handleResult(Serializable externalProcessId, WorkItem workItem, WorkflowSession session, MetaDataMap arguments) throws WorkflowException;

}
