/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 2011 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
package com.day.crx.packaging.gfx;

import java.awt.Dimension;
import java.awt.Rectangle;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.Writer;
import java.util.HashMap;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.RepositoryException;

import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.jackrabbit.util.Text;

import com.day.image.Layer;

/**
 * Provides convenience methods for displaying images.
 */
public class ImageResource extends DownloadResource {

    /**
     * name of the html width property
     */
    public static final String PN_HTML_WIDTH = "htmlWidth";

    /**
     * name of the html height property
     */
    public static final String PN_HTML_HEIGHT = "htmlHeight";

    /**
     * name of the width property
     */
    public static final String PN_WIDTH = "width";

    /**
     * name of the height property
     */
    public static final String PN_HEIGHT = "height";

    /**
     * name of the minimal width property. used for resizing.
     */
    public static final String PN_MIN_WIDTH = "minWidth";

    /**
     * name of the minimal height property. used for resizing.
     */
    public static final String PN_MIN_HEIGHT = "minHeight";

    /**
     * name of the maximal width property. used for resizing.
     */
    public static final String PN_MAX_WIDTH = "maxWidth";

    /**
     * name of the maximal height property. used for resizing.
     */
    public static final String PN_MAX_HEIGHT = "maxHeight";

    /**
     * name of the alt name property
     */
    public static final String PN_ALT = "alt";

    /**
     * name of the image crop property
     */
    public static final String PN_IMAGE_CROP = "imageCrop";

    /**
     * name of the image rotation property
     */
    public static final String PN_IMAGE_ROTATE = "imageRotate";

    /**
     * name of the link URL property
     */
    public static final String PN_LINK_URL = "linkURL";

    /**
     * Extension from type.
     */
    private String extFromType;

    /**
     * Creates a new image based on the given resource. the image properties are
     * considered to 'on' the given resource.
     *
     * @param node resource of the image
     * @throws IllegalArgumentException if the given resource is not adaptable to node.
     */
    public ImageResource(Node node) {
        super(node);
        super.setExtension(".png");

        // init suffix with last mod date to avoid browser caching issues.
        // download class initializes it otherwise with the filename - which is not desirable
        try {
            String suffix = "";
            if(node != null) {
                long lastMod = 0;
                if (node.hasProperty(Property.JCR_LAST_MODIFIED)) {
                    lastMod = node.getProperty(Property.JCR_LAST_MODIFIED).getLong();
                } else if (node.hasProperty(Property.JCR_CREATED)) {
                    lastMod = node.getProperty(Property.JCR_CREATED).getLong();
                }
                long fileLastMod = 0;
                if (getFileReference().length() > 0) {
                    try {
                        Node refNode = getReferencedNode(getFileReference());
                        if (refNode != null) {
                            if (refNode.getNode(Property.JCR_CONTENT).hasProperty(Property.JCR_LAST_MODIFIED)) {
                                fileLastMod = refNode.getNode(Property.JCR_CONTENT).getProperty(Property.JCR_LAST_MODIFIED).getLong();
                            } else if (refNode.hasProperty(Property.JCR_CREATED)) {
                                fileLastMod = refNode.getProperty(Property.JCR_CREATED).getLong();
                            }
                        }
                    }
                    catch (Exception e) {
                        // e.g. asset not found; use lastMod
                    }
                }
                if (fileLastMod > lastMod) {
                    lastMod = fileLastMod;
                }
                if (lastMod != 0) {
                    suffix += lastMod;
                    suffix += getExtension();
                }
            }
            setSuffix(suffix);
        } catch (RepositoryException re) {
            // ignore
        }
    }

    /**
     * Creates a new image based on the given resource. the image properties are
     * considered to 'on' the given resource unless <code>imageName</code>
     * is specified. then the respective child resource holds the image
     * properties.
     *
     * @param resource  current resource
     * @param imageName name of the image resource
     * @throws IllegalArgumentException if the given resource is not adaptable to node.
     */
    public ImageResource(Node resource, String imageName) {
        this(getRelativeResource(resource, imageName));
    }

    /**
     * Returns the relative resource or a null
     *
     * @param resource "parent" resource
     * @param relPath relative path
     * @return the resource
     */
    protected static Node getRelativeResource(Node resource, String relPath) {
        if (relPath == null) {
            return resource;
        }
        try {
            return resource.getNode(relPath);
        } catch (RepositoryException e) {
            return null;
        }
    }

    /**
     * Returns the image alt name as defined by the {@value #PN_ALT}
     * or overridden by {@link #setAlt(String)}.
     *
     * @return the alt name
     * @see #PN_ALT
     */
    public String getAlt() {
        String alt = get(getItemName(PN_ALT));
        if (alt.length() == 0) {
            alt = getTitle();
        }
        return alt.length() == 0 ? getFileNodePath(): alt;
    }

    /**
     * Sets the alt name.
     * @param alt the alt name.
     */
    public void setAlt(String alt) {
        set(PN_ALT, alt);
    }

    /**
     * Returns the source attribute of this image. the source is computed as
     * follows:
     * <ul>
     * <li> if a selector is defined the path of the current resource concatenated
     *      with the selector, extension, and suffix is used.
     * <li> if a file node path is defined it is concatenated with the extension and suffix.
     * <li> if a file reference is defined it is concatenated with the extension and suffix.
     * </ul>
     *
     * @return the source attribute
     */
    public String getSrc() {
        return getHref();
    }

    /**
     * Tries to calculate the extension from the mime-type of the underlying
     * image.
     * @return the mime-type dependant extension or ".png" if not determinable
     */
    @Override
    public String getExtension() {
        if (extFromType == null) {
            try {
                extFromType = ImageHelper.getExtensionFromType(getMimeType());
            } catch (RepositoryException e) {
                // ignore
            }
            if (extFromType == null) {
                extFromType = super.getExtension();
            } else if (!extFromType.startsWith(".")) {
                extFromType = "." + extFromType;
            }
        }
        return extFromType;
    }

    @Override
    public void setExtension(String extension) {
        extFromType = extension;
        super.setExtension(extension);
    }

    /**
     * Sets the source attribute
     * @param src the source attribute
     */
    public void setSrc(String src) {
        super.setHref(src);
    }

    /**
     * Writes this image as tag to the given writer by invoking
     * {@link #doDraw(PrintWriter)} if {@link #canDraw()} returns <code>true</code>.
     *
     * @param w the writer
     * @throws IOException if an I/O error occurs
     */
    public void draw(Writer w) throws IOException {
        if (canDraw()) {
            doDraw(new PrintWriter(w));
        }
    }

    /**
     * Writes this image as tag to the given writer by invoking the following
     * - calls {@link #getImageTagAttributes()}
     * - prints the link tag if needed
     * - prints the image tag
     * - prints the attributes
     * - closes the image tag
     * - closes the link tag
     *
     * @param out the writer
     */
    protected void doDraw(PrintWriter out) {
        Map<String, String> attributes = getImageTagAttributes();
        String linkURL = get(PN_LINK_URL);
        if (linkURL.length() > 0) {
            out.printf("<a href=\"%s\">", completeHREF(linkURL));
        }
        out.print("<img ");
        for (Map.Entry<String, String> e : attributes.entrySet()) {
            Object value = e.getValue();
            out.printf("%s=\"%s\" ",
                    StringEscapeUtils.escapeHtml4(e.getKey()),
                    StringEscapeUtils.escapeHtml4(value != null ? value.toString() : ""));
        }
        out.print(">");

        if (linkURL.length() > 0) {
            out.print("</a>");
        }
    }

    /**
     * checks if this image can be drawn.
     * @return <code>true</code> if it can be drawn
     */
    protected boolean canDraw() {
        return hasContent();
    }

    /**
     * Collects the image tag attributes.
     * @return the attributes
     */
    protected Map<String, String> getImageTagAttributes() {
        Map<String, String> attributes = new HashMap<String, String>();
        if (get(getItemName(PN_HTML_WIDTH)).length() > 0) {
            attributes.put("width", get(getItemName(PN_HTML_WIDTH)));
        }
        if (get(getItemName(PN_HTML_HEIGHT)).length() > 0) {
            attributes.put("height", get(getItemName(PN_HTML_HEIGHT)));
        }
        String src = getSrc();
        if (src != null) {
            attributes.put("src", Text.escape(src, '%', true));
        }
        attributes.put("alt", getAlt());
        attributes.put("title", getTitle());
        if (attrs != null) {
            attributes.putAll(attrs);
        }
        return attributes;
    }

    /**
     * Completes the href with the same formatting than link into RTE
     * @param href the href
     * @return the completed href
     */
    private String completeHREF(String href) {
        if (href != null && href.length() > 0) {
            //only for internal links
            if ((href.charAt(0) == '/') || (href.charAt(0) == '#')) {
                int anchorPos = href.indexOf("#");
                if (anchorPos == 0) {
                    // change nothing if we have an "anchor only"-HREF
                    return href;
                }
                String anchor = "";
                if (anchorPos > 0) {
                    anchor = href.substring(anchorPos, href.length());
                    href = href.substring(0, anchorPos);
                }

                // add extension to href if necessary
                int extSepPos = href.lastIndexOf(".");
                int slashPos = href.lastIndexOf("/");
                if ((extSepPos <= 0) || (extSepPos < slashPos)) {
                    href = Text.escape(href, '%', true) + ".html" + anchor;
                }
            }
        }
        return href;
    }

    /**
     * Returns the cropping rectangle as defined by the {@value #PN_IMAGE_CROP}.
     *
     * @return the cropping rectangle or <code>null</code>
     */
    public Rectangle getCropRect() {
        String cropData = get(getItemName(PN_IMAGE_CROP));
        if (cropData.length() > 0) {
            return ImageHelper.getCropRect(cropData, getPath());
        }
        return null;
    }

    /**
     * Returns the rotation angle as defined by the {@value #PN_IMAGE_ROTATE}.
     *
     * @return the rotation angle (in degrees)
     */
    public int getRotation() {
        String rotation = get(getItemName(PN_IMAGE_ROTATE));
        if (rotation.length() > 0) {
            try {
                return Integer.parseInt(rotation);
            } catch (NumberFormatException nfe) {
                // use default return of 0
            }
        }
        return 0;
    }

    /**
     * Resizes the given layer according to the dimensions defined in this image.
     * See {@link ImageHelper#resize(Layer, Dimension, Dimension, Dimension)}
     * for more details about the resizing algorithm.
     *
     * @param layer the layer to resize
     * @return the layer or <code>null</code> if the layer is untouched
     */
    public Layer resize(Layer layer) {
        Dimension d = new Dimension(
                get(getItemName(PN_WIDTH), 0),
                get(getItemName(PN_HEIGHT), 0));
        Dimension min = new Dimension(
                get(getItemName(PN_MIN_WIDTH), 0),
                get(getItemName(PN_MIN_HEIGHT), 0));
        Dimension max = new Dimension(
                get(getItemName(PN_MAX_WIDTH), 0),
                get(getItemName(PN_MAX_HEIGHT), 0));
        return ImageHelper.resize(layer, d, min, max);
    }

    /**
     * Crops the layer using the internal crop rectangle. if the crop rectangle
     * is empty no cropping is performed and <code>null</code> is returned.
     *
     * @param layer the layer
     * @return cropped layer or <code>null</code>
     */
    public Layer crop(Layer layer) {
        Rectangle rect = getCropRect();
        if (rect != null) {
            layer.crop(rect);
            return layer;
        }
        return null;
    }

    /**
     * Rotates the layer using the internal rotation angle. If no rotation other than
     * 0 is defined, no rotation is performed and <code>null</code> is returned.
     *
     * @param layer the layer
     * @return cropped layer or <code>null</code>
     */
    public Layer rotate(Layer layer) {
        int rotation = getRotation();
        if (rotation != 0) {
            layer.rotate(rotation);
            return layer;
        }
        return null;
    }

    /**
     * Returns the layer addressed by this image.
     *
     * @param cropped apply cropping if <code>true</code>
     * @param resized apply resizing if <code>true</code>
     * @param rotated apply rotation if <code>true</code>
     * @return the layer
     * @throws IOException         if an I/O error occurs.
     * @throws RepositoryException if a repository error occurs.
     */
    public Layer getLayer(boolean cropped, boolean resized, boolean rotated)
            throws IOException, RepositoryException {
        Layer layer = null;
        Property data = getData();
        if (data != null) {
            layer = ImageHelper.createLayer(data);
            if (layer != null && cropped) {
                crop(layer);
            }
            if (layer != null && resized) {
                resize(layer);
            }
            if (layer != null && rotated) {
                rotate(layer);
            }
        }
        return layer;
    }
}