/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 2017 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
package com.day.crx.packaging.validation;

import java.io.File;
import java.io.IOException;
import java.util.List;

import javax.jcr.RepositoryException;
import javax.jcr.Session;

import com.day.crx.packaging.validation.impl.AclWarning;
import org.apache.jackrabbit.vault.packaging.JcrPackage;

import com.day.crx.packaging.validation.impl.OverlayRebaseWarning;
import com.day.crx.packaging.validation.impl.UnsatisfiedImport;

/**
 * <code>ContentPackageValidator</code>
 * Validates a content package before installing it in /crx/packmgr. Reports all unsatisfied OSGi 
 * package imports, overlay rebase warnings which will occur if the content package is installed.
 */
public interface ContentPackageValidator {
    
    /**
     * Validates a content package for unsatisfied OSGi package imports, given its <code>JcrPackage</code> object.
     * @param <code>JcrPackage</code> object of the content package
     * @return List of unsatisfied OSGi package imports
     * @throws IOException
     * @throws RepositoryException
     */
    List<UnsatisfiedImport> validateOsgiImports(JcrPackage jcrPackage) throws RepositoryException, IOException;
    
    /**
     * Validates a content package for unsatisfied OSGi package imports, given its <code>File</code> object.
     * @param <code>File</code> object of the content package
     * @return List of unsatisfied OSGi package imports
     * @throws IOException
     */
    List<UnsatisfiedImport> validateOsgiImports(File contentPackageFile) throws IOException;
    
    /**
     * Validates a content package for overlay rebase warnings, given its <code>JcrPackage</code> object.
     * @param <code>JcrPackage</code> object of the content package
     * @param <code>Session</code> to get access to JCR 
     * @return List of overlay rebase warnings
     * @throws IOException 
     * @throws RepositoryException 
     */
    List<OverlayRebaseWarning> validateOverlays(JcrPackage pack, Session session) throws RepositoryException, IOException;
    
    /**
     * Validates a content package for overlay rebase warnings, given its <code>File</code> object.
     * @param <code>File</code> object of the content package 
     * @param <code>Session</code> to get access to JCR
     * @return List of overlay rebase warnings
     * @throws IOException
     * @throws RepositoryException 
     */
    List<OverlayRebaseWarning> validateOverlays(File contentPackageFile, Session session) throws IOException, RepositoryException;

    /**
     * Validates a content package for ACL warnings, given its <code>JcrPackage</code> object.
     * @param jcrPackage <code>JcrPackage</code> object of the content package
     * @param session <code>Session</code> to get access to JCR
     * @return List of ACL warnings
     * @throws IOException
     * @throws RepositoryException
     */
    List<AclWarning> validateAcls(JcrPackage jcrPackage, Session session) throws IOException, RepositoryException;

    /**
     * Validates a content package for ACL warnings, given its <code>File</code> object.
     * @param contentPackageFile <code>File</code> object of the content package
     * @param session <code>Session</code> to get access to JCR
     * @return List of ACL warnings
     * @throws IOException
     * @throws RepositoryException
     */
    List<AclWarning> validateAcls(File contentPackageFile, Session session) throws IOException, RepositoryException;
}