/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.crx.security.token;

import com.adobe.granite.crypto.CryptoException;
import com.day.crx.security.token.impl.TokenAuthenticationHandler;
import org.apache.jackrabbit.api.security.authentication.token.TokenCredentials;
import org.apache.sling.auth.core.spi.AuthenticationInfo;
import org.apache.sling.jcr.api.SlingRepository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * The <code>TokenUtil</code> class provides utility methods to generate login-tokens
 */
public class TokenUtil {
    /**
     * Private constructor to avoid instantiation
     */
    private TokenUtil() {
    }

    /**
     * The AUTH_TYPE of the {@link com.day.crx.security.token.impl.TokenAuthenticationHandler}
     */
    private static final String AUTH_TYPE = "TOKEN";

    /*
     * The name of the token attribute to be set on {@link javax.jcr.SimpleCredentials}
     */
    private static final String TOKEN_ATTRIBUTE = ".token";

    /**
     * The name of the credentials attribute in {@link AuthenticationInfo}
     */
    private static final String JCR_CREDENTIALS_ATTRIBUTE = "user.jcr.credentials";

    /**
     * Generates a login-token for a given userId, sets the token-cookie, and returns an
     * {@link AuthenticationInfo} object suited for the {@link com.day.crx.security.token.impl.TokenAuthenticationHandler}
     *
     * @param request    The {@link HttpServletRequest} associated with this authentication attempt
     * @param response   The {@link HttpServletResponse} on which to set the cookie
     * @param repository The {@link SlingRepository} to authenticate against
     * @param userId     The ID of the authenticating user
     * @param httpOnly   Whether or not to set the <code>HttpOnly</code> attribute on the token cookie
     * @return An AuthenticationInfo of type {@link #AUTH_TYPE}
     * @throws RepositoryException
     */
    public static AuthenticationInfo createCredentials(HttpServletRequest request, HttpServletResponse response, SlingRepository repository, String userId, boolean httpOnly) throws RepositoryException {
        Session adminSession = null;
        Session userSession = null;
        try {
            // generate token for userId through impersonation
            adminSession = repository.loginAdministrative(null);
            AuthenticationInfo authInfo = new AuthenticationInfo(AUTH_TYPE, userId);
            boolean encapsulatedToken = TokenAuthenticationHandler.isEncapsulatedToken();
            if (encapsulatedToken) {
                String token = TokenAuthenticationHandler.buildEncapsulatedToken(userId);
                TokenCredentials tc = new TokenCredentials(token);
                authInfo.put(JCR_CREDENTIALS_ATTRIBUTE, tc);
                TokenCookie.update(request, response, getRepositoryId(encapsulatedToken),tc.getToken(),
                        adminSession.getWorkspace().getName(), httpOnly); 
            } else {
                SimpleCredentials sc = new SimpleCredentials(userId, new char[0]);
                sc.setAttribute(TOKEN_ATTRIBUTE, "");
                userSession = adminSession.impersonate(sc);

                // create authentication info from token credentials
                TokenCredentials tc = new TokenCredentials((String) sc.getAttribute(TOKEN_ATTRIBUTE));                
                authInfo.put(JCR_CREDENTIALS_ATTRIBUTE, tc);

                // set/update login token cookie
                TokenCookie.update(request, response, getRepositoryId(encapsulatedToken), tc.getToken(),
                        adminSession.getWorkspace().getName(), httpOnly);
            }          
            
            return authInfo;
        } catch (RepositoryException e) {
            throw new RepositoryException(
                    "Failed to generate login-token: Could not access Repository",
                    e);
        } catch (CryptoException e) {
            throw new RepositoryException(
                    "Failed to generate login-token",
                    e);
        } finally {
            if (userSession != null) {
                userSession.logout();
            }
            if (adminSession != null) {
                adminSession.logout();
            }
        }
    }

    private static String getRepositoryId(boolean encapsulatedToken) {
        return TokenAuthenticationHandler.getRepositoryId(encapsulatedToken);
    }
}
