/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.durbo;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

import javax.jcr.PropertyType;
import javax.jcr.RepositoryException;
import javax.jcr.Value;
import javax.jcr.ValueFactory;

import org.apache.commons.io.IOUtils;

/**
 * <code>DurboValue</code> implements an internal representation of a value
 * retrieved from the DurboInput. It is either based on bytes or on a stream.
 */
public class DurboValue {

    /**
     * the destination type
     */
    private final int type;

    /**
     * the internal byte buffer. or null, if based on a stream.
     */
    private byte[] bytes;

    /**
     * the internal stream. or null, if based on a bytes array
     */
    private InputStream in;

    /**
     * the size of this value
     */
    private long size;

    /**
     * Creates a new value that is based on a byte array
     *
     * @param type  the property type
     * @param bytes the byte array
     * @see PropertyType
     */
    public DurboValue(int type, byte[] bytes) {
        this.type = type;
        this.bytes = bytes;
        this.size = bytes.length;
    }

    /**
     * Creates a new value that is based on a stream
     *
     * @param type the property type
     * @param in   the input stream
     * @param size the size of the data in the input stream
     * @see PropertyType
     */
    public DurboValue(int type, InputStream in, long size) {
        this.type = type;
        this.in = in;
        this.size = size;
    }

    /**
     * Returns the property type of this value.
     *
     * @return the property type.
     * @see PropertyType
     */
    public int getType() {
        return type;
    }

    /**
     * Returns the size of this value
     *
     * @return the size of this value
     * @deprecated since 2.1.2 use {@link #getLength()} instead.
     */
    public int getSize() {
        if (size > (long) Integer.MAX_VALUE) {
            throw new IllegalArgumentException("Size greater than 2^31. Use getLength() instead.");
        }
        return (int) size;
    }

    /**
     * Returns the size of this value
     *
     * @return the size of this value
     */
    public long getLength() {
        return size;
    }

    
    /**
     * Returns a byte representation of this value. If this value initially
     * is based on an input stream, it's content are read into a new byte buffer
     * and the stream is closed.
     *
     * @return the byte array of this value
     * @throws IOException if an I/O error during spooling occurs.
     */
    public byte[] getBytes() throws IOException {
        if (bytes == null) {
            bytes = IOUtils.toByteArray(in);
            in.close();
            in = null;
        }
        return bytes;
    }

    /**
     * Returns a string representation of this value. the bytes retrieved by
     * {@link #getBytes()} are converted in a string using utf-8 as encoding.
     *
     * @return the string representation
     * @throws IOException if an I/O error during conversion occurs.
     */
    public String getString() throws IOException {
        return new String(getBytes(), "utf-8");
    }

    /**
     * Returns a double representation of this value. the string retrieved by
     * {@link #getString()} is converted to a double using
     * {@link Double#parseDouble(String)}.
     *
     * @return the double value
     * @throws IOException if an i/O error during conversion occurs.
     */
    public double getDouble() throws IOException {
        return Double.parseDouble(getString());
    }

    /**
     * Returns a stream representation of this value. if this value originally
     * was based on a stream, this stream is returned otherwise a new
     * {@link ByteArrayInputStream} of the internal byte array is returned.
     *
     * @return a stream on this value
     */
    public InputStream getStream() {
        if (in == null) {
            return new ByteArrayInputStream(bytes);
        } else {
            return in;
        }
    }

    /**
     * Checks if this value internally is based on a byte array.
     *
     * @return <code>true</code> if this value is based on a byte array;
     *         <code>false</code> otherwise.
     */
    public boolean isByteArrayBased() {
        return bytes != null;
    }

    /**
     * Returns a new jcr value from this value using the value type. If this
     * value is of type {@link PropertyType#BINARY} the new value is
     * created using {@link #getStream()}, otherwise the new value is
     * created using {@link #getString()}.
     *
     * @param factory the value factory that is used to create the value
     * @return the new jcr value
     * @throws IOException         if an I/O error occurs.
     * @throws RepositoryException if a repository error occurs.
     */
    public Value toJcrValue(ValueFactory factory)
            throws IOException, RepositoryException {
        if (type == PropertyType.BINARY) {
            return factory.createValue(factory.createBinary(getStream()));
        } else {
            return factory.createValue(getString(), type);
        }
    }
}