/*
 * $Id: BufferedRAFInputStream.java 12345 2004-08-22 04:56:09Z fielding $
 *
 * Copyright 1997-2004 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.io;

import java.io.InputStream;
import java.io.IOException;
import java.io.RandomAccessFile;

/**
 * This class implements a buffered inputstream on a random access file.
 *
 * @version $Revision: 1.2 $
 * @author tripod
 * @since echidna
 */
public class BufferedRAFInputStream extends InputStream {

    /** the buffer */
    private final byte[] buffer = new byte[8192];

    /** the raf */
    private RandomAccessFile raf;

    /** the starting position of the buffer in the raf */
    private long bufferStart;

    /** the read position in the buffer */
    private int bufferPos;

    /** the end of valid data in the buffer */
    private int bufferEnd;

    /** dummy buffer for {@link #read()} */
    private byte[] one = new byte[1];

    /**
     * Creates a new <code>BufferedRafInputStream</code> on top of a random
     * access file
     *
     * @param file The RAF
     */
    public BufferedRAFInputStream(RandomAccessFile file) throws IOException {
        raf = file;
        bufferStart=raf.getFilePointer();
    }

    /**
     * @see java.io.InputStream#read()
     */
    public int read() throws IOException {
        // lazy
        int r = read(one, 0, 1);
        return r>=0 ? one[0] : -1;
    }

    /**
     * @see java.io.InputStream#read(byte[])
     */
    public int read(byte b[]) throws IOException {
        return read(b, 0, b.length);
    }

    /**
     * @see java.io.InputStream#read(byte[], int, int)
     */
    public int read(byte b[], int off, int len) throws IOException {
        if (bufferEnd<0) {
            return -1;
        }
        int total=0;
        while (total<len && bufferEnd>=0) {
            int toCopy = Math.min(bufferEnd-bufferPos, len-total);
            System.arraycopy(buffer, bufferPos, b, off, toCopy);
            bufferPos+=toCopy;
            total+=toCopy;
            off+=toCopy;
            if (bufferPos==bufferEnd) {
                bufferStart = raf.getFilePointer();
                bufferEnd = raf.read(buffer);
                bufferPos = 0;
            }
        }
        return total;
    }

    /**
     * @see java.io.InputStream#skip(long)
     */
    public long skip(long n) throws IOException {
        if (bufferEnd < 0) {
            return -1;
        }
        int len = bufferEnd-bufferPos;
        if (n < len) {
            bufferPos+=n;
            return n;
        }

        int sk = raf.skipBytes((int) (n - len) );
        bufferPos=bufferEnd=0;
        bufferStart=raf.getFilePointer();
        return sk + len;
    }

    /**
     * Seeks to the indicated position
     * @param n
     * @throws java.io.IOException
     */
    public void seek(long n) throws IOException {
        bufferPos = (int) (n-bufferStart);
        if (bufferPos<0 || bufferPos>bufferEnd) {
            bufferStart = n;
            bufferPos = 0;
            bufferEnd = 0;
            raf.seek(n);
        }
    }

    /**
     * Returns the position of the filepointer
     * @return the position of the filepointer
     * @throws java.io.IOException
     */
    public long getFilePointer() throws IOException {
        return bufferStart + bufferPos;
    }

    /**
     * @see java.io.InputStream#available()
     */
    public int available() throws IOException {
        return (int) (raf.length() - (bufferStart + bufferPos));
    }

    /**
     * @see java.io.InputStream#close()
     */
    public void close() throws IOException {
        raf=null;
    }
}