
/**
 * $Id: FileUtils.java 12345 2004-08-22 04:56:09Z fielding $
 *
 * Copyright 1997-2004 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 *
 * @version $Revision: 1.7 $, $Date: 2004-08-22 06:56:09 +0200 (Sun, 22 Aug 2004) $
 */

package com.day.io.file;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedWriter;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A collection of static methods useful for file related activities -
 * this is almost identical to the file com.day.cq.jsp.tag.util.FileUtils
 */
public class FileUtils {
    private static final Logger log =
        LoggerFactory.getLogger(FileUtils.class);

    private static byte[] readFirstXBytesFromFile(File fileName, int iNoOfBytes) throws IOException {
        log.debug("Starting readFirstXBytesFromFile ("+""+"fileName : "+fileName + " , "+"iNoOfBytes : "+iNoOfBytes + " , "+")");
        byte[] buffer = new byte[ iNoOfBytes ];
        FileInputStream in = new FileInputStream(fileName);
        in.read(buffer);
        in.close();
        return buffer;
    }


    public static void writeBytesToFile(byte[] data, File fOutput) throws IOException {
        log.debug("Starting writeBytesToFile ("+""+"data : "+data + " , "+"fOutput : "+fOutput + " , "+")");
        BufferedOutputStream out = null;
        try {
            out = new BufferedOutputStream(new FileOutputStream(fOutput));
            out.write(data);
            out.flush();
        } finally {
            if (out != null) { out.close(); }
        }
    }

    /**
     * Retrieve a file from the root of the classes directory.
     *
     * @todo Test for JAR and WAR cases
     */
    public static File getFileFromClassesRoot(String fileName) throws FileNotFoundException {
        log.debug("Starting getFileFromClassesRoot ("+""+"fileName : "+fileName + " , "+")");
        return getFileFromClassesRoot(fileName, FileUtils.class);
    }

    /**
     * Retrieve a file from the root of the classes directory.
     *
     * @todo Test for JAR and WAR cases
     */
    public static File getFileFromClassesRoot(String fileName, Class relativeTo) throws FileNotFoundException {
        log.debug("Starting getFileFromClassesRoot ("+""+"fileName : "+fileName + " , "+"relativeTo : "+relativeTo + " , "+")");
        URL u = relativeTo.getResource("/resources/"+fileName);
        if ((u==null) || (u.getFile()==null)) {
            throw new FileNotFoundException("File "+fileName+" not found in the root of the classes directory "+
                    "by com.day.io.file.FileUtils - looking at "+relativeTo.getResource("/").getFile()+"/resources/");
        }
        File f = new File(u.getFile());
        return f;
    }


    public static InputStream getStreamFromClassesRoot(String fileName) throws FileNotFoundException {
        log.debug("Starting getStreamFromClassesRoot ("+""+"fileName : "+fileName + " , "+")");
        return getStreamFromClassesRoot(fileName, FileUtils.class);
    }

    public static InputStream getStreamFromClassesRoot(String fileName, Class relativeTo) throws FileNotFoundException {
        log.debug("Starting getStreamFromClassesRoot ("+""+"fileName : "+fileName + " , "+"relativeTo : "+relativeTo + " , "+")");
        while (fileName.startsWith("/")) {
            fileName = fileName.substring(1);
        }
        InputStream in = relativeTo.getResourceAsStream("/resources/" + fileName);
        if (in==null) {
            try {
                throw new FileNotFoundException("File "+fileName+" not found in the root of the classes directory "+
                        "by com.day.io.file.FileUtils - looking at "+relativeTo.getResource("/").getFile()+"/resources/");
            }
            catch (NullPointerException e) {
                throw new FileNotFoundException("File "+fileName+" not found in the root of the classes directory "+
                        "by com.day.io.file.FileUtils - looking at "+relativeTo.getName() + "'s class file.");
            }
        }
        return in;
    }


    public static byte[] getBytesFromClassesRoot(String fileName) throws IOException {
        log.debug("Starting getBytesFromClassesRoot ("+""+"fileName : "+fileName + " , "+")");
        return StreamUtils.readBytesFromStream(FileUtils.getStreamFromClassesRoot(fileName));
    }

    public static String getStringFromClassesRoot(String fileName) throws IOException {
        log.debug("Starting getStringFromClassesRoot ("+""+"fileName : "+fileName + " , "+")");
        return StreamUtils.convertStreamToString(FileUtils.getStreamFromClassesRoot(fileName));
    }

    /**
     * Concatenate two byte arrays.
     */
    private static byte[] append(byte[] source, byte[] addition, int length) {
        log.debug("Starting append ("+""+"source : "+source + " , "+"addition : "+addition + " , "+"length : "+length + " , "+")");
        byte[] dest = new byte[source.length + length];
        System.arraycopy(source, 0, dest, 0, source.length);
        System.arraycopy(addition, 0, dest, source.length, length);
        return dest;
    }

    public static byte[] readBytesFromStream(InputStream in) throws IOException {
        log.debug("Starting readBytesFromStream ("+""+"in : "+in + " , "+")");
        // 8k buffer on the stream
        BufferedInputStream bis = new BufferedInputStream(new DataInputStream(in), 8192);

        int bytesread = 0;
        int firstbyte = 0;
        byte[] content = new byte[1];

        // This is needed in order that bis.available() returns a sensible value.  Almost like
        // we need to 'kick' the stream to get it to work correctly.
        content[0] = (byte)bis.read();

        // A good initial size.
        int BUFFER_SIZE = bis.available() + 1;  // We've already read one byte

        byte[] readBuffer = new byte[BUFFER_SIZE];
        while ((bytesread = bis.read(readBuffer, 0, readBuffer.length)) > 0) {

            // Append the new content to the existing, and loop
            content = append(content, readBuffer, bytesread);
        }
        bis.close();

        return content;
    }

    public static byte[] readBytesFromFile(File f) throws IOException {
        log.debug("Starting readBytesFromFile ("+""+"f : "+f + " , "+")");
        FileInputStream in = new FileInputStream(f);
        return readBytesFromStream(in);
    }

    public static void writeStringToFile(String strData, File fOutput) throws IOException {
        log.debug("Starting writeStringToFile ("+""+"strData : "+strData + " , "+"fOutput : "+fOutput + " , "+")");
        BufferedWriter bw = null;
        try {
            bw = new BufferedWriter(new FileWriter(fOutput));
            bw.write(strData);
        } finally {
            if (bw != null) {
                bw.close();
            }
        }
    }


    public static String readFileToString(File fInput) throws IOException {
        log.debug("Starting readFileToString ("+""+"fInput : "+fInput + " , "+")");
        String strResult = null;
        FileReader in = new FileReader(fInput);
        int iSize = (int) fInput.length();
        char[] caData = new char[iSize];
        int iCharsRead = 0;
        while (iCharsRead < iSize) {
            iCharsRead += in.read(caData, iCharsRead, iSize - iCharsRead);
        }
        in.close();
        strResult = new String(caData);
        return strResult;
    }

    public static boolean isSame(byte[] contents, File toCheck) throws IOException {
        log.debug("Starting isSame ("+""+"contents : "+contents + " , "+"toCheck : "+toCheck + " , "+")");

        byte[] buf = new byte[(int)(toCheck.length())];
        DataInputStream in = new DataInputStream(new FileInputStream(toCheck));

        int val = 0;
        int pos = 0;
        while ((val = in.read()) != -1) {
            buf[pos] = (byte)val;
            pos++;
        }
        in.close();

        if (contents.length != buf.length) return false;
        for (int i=0; i<contents.length; i++) {
            if (contents[i] != buf[i]) return false;
        }
        return true;
    }

    /**
     * Copy a file.
     */
    public static synchronized void copyFile(File src, File dest) throws IOException {
        log.debug("Starting copyFile ("+""+"src : "+src + " , "+"dest : "+dest + " , "+")");

        // First make sure the source file exists, is a file, and is readable.
        if (!src.exists()) {
            throw new IOException("FileCopy: no such source file: " + src);
        }
        if (!src.isFile()) {
            throw new IOException("FileCopy: cannot copy directory: " + src);
        }
        if (!src.canRead()) {
            throw new IOException("FileCopy: source file is unreadable: " + src);
        }

        // If the destination is a directory, use the source file name as the destination file name
        if (dest.isDirectory()) {
            dest = new File(dest, src.getName());
        }

        // If the destination exists, make sure it is a writeable file.  If the destination doesn't
        // exist, make sure the directory exists and is writeable.
        if (dest.exists()) {
            if (!dest.canWrite()) {
                throw new IOException("FileCopy: destination file is unwriteable: " + dest);
            }
        } else {
            // If file doesn't exist, check if directory exists and is writeable. If getParent() returns null,
            // then the directory is the current directory, so look up the user.dir system property to find
            // out what that is.
            String parent = dest.getParent();
            if (parent == null) {
                parent = System.getProperty("user.dir");
            }

            File dir = new File(parent);

            if (!dir.exists()) {
                throw new IOException("FileCopy: destination directory doesn't exist: " + parent);
            }
            if (dir.isFile()) {
                throw new IOException("FileCopy: destination is not a directory: " + parent);
            }
            if (!dir.canWrite()) {
                throw new IOException("FileCopy: destination directory is unwriteable: " + parent);
            }
        }

        FileInputStream  from = null;
        FileOutputStream to   = null;

        try {
            from = new FileInputStream(src);
            to   = new FileOutputStream(dest);

            byte[] buffer = new byte[4096];
            int bytes_read;

            while ((bytes_read = from.read(buffer)) != -1) {
                to.write(buffer, 0, bytes_read);
            }
        }

         finally {
            try { from.close(); } catch (Exception e) { ; }
            try { to.close(); } catch (IOException e) { ; }
        }
    }
}
