
/**
 * $Id: TempDirectory.java 12345 2004-08-22 04:56:09Z fielding $
 *
 * Copyright 1997-2004 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 *
 * @version $Revision: 1.7 $, $Date: 2004-08-22 06:56:09 +0200 (Sun, 22 Aug 2004) $
 */

package com.day.io.file;

import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Encapsulates a temporary directory, that should work cross-platform
 * and on JDK 1.1. This is mostly useful for test code.
 * <p>
 * Note the finalize method, which should delete the directory when
 * this object is garbage collected.
 */
public class TempDirectory {
    private static final Logger log =
        LoggerFactory.getLogger(TempDirectory.class);
    private File tempDir = null;

    public TempDirectory(String name) throws IOException {
        createTempDir(name);
    }

    private void createTempDir(String name) throws IOException {
        log.debug("Starting createTempDir ("+""+"name : "+name + " , "+")");
        tempDir = new File(System.getProperty("java.io.tmpdir"), "tmp"+name);
        int i=0;
        while (tempDir.exists()) {
            i++;
            tempDir = new File(System.getProperty("java.io.tmpdir"), "tmp"+name+Integer.toString(i));
        }
//        if (tempDir.exists()) {
//                throw new IOException("Could not make temp directory "+tempDir.getPath()+" - it already exists");
//        }
        if (!tempDir.mkdir()) { throw new IOException("Could not make temp directory "+tempDir.getPath()); }
        empty();
    }

    /**
     * @todo This should be recursive
     */
    public void empty() throws IOException {
        log.debug("Starting empty");
        File[] files = listFiles();
        for (int i=0; i<files.length; i++) {
            if (!files[i].delete()) { throw new IOException("Cannot delete file "+files[i].getPath()+" in the temp directory");  }
        }
        if (!isEmpty()) { throw new IOException("Could not delete all files in the temp directory "+tempDir.getPath()); }
    }

    /**
     * This is taken from the JDK 1.2 File class
     */
    public URL toURL() throws MalformedURLException {
        log.debug("Starting toURL");
        // In JDK 1.2, the following method can be replaced by
        //      return tempDir.toURL()
        String path = tempDir.getAbsolutePath();
        if (File.separatorChar != '/') {
            path = path.replace(File.separatorChar, '/');
        }
        if (!path.startsWith("/")) {
            path = "/" + path;
        }
        if (!path.endsWith("/") && tempDir.isDirectory()) {
            path = path + "/";
        }
        return new URL("file", "", path);
    }



    public File[] listFiles() {
        log.debug("Starting listFiles");
        // In JDK 1.2, the following block can be replaced by
        //      return tmpDir.listFiles();
        String[] fileNames = tempDir.list();
        File[] files = new File[fileNames.length];
        for (int i=0; i<files.length; i++) {
            files[i] = new File(tempDir, fileNames[i]);
        }
        return files;
    }

    /**
     * This is useful for tests - specifically the one in com.day.io.htmlparser.TextContentListener,
     * which creates files, downloads them, and then checks them.
     */
    public File createTempFile(String fileName) throws IOException {
        log.debug("Starting createTempFile ("+""+"fileName : "+fileName + " , "+")");
        File tempFile = new File(tempDir, fileName);
        FileUtils.writeStringToFile(fileName, tempFile);
        return tempFile;
    }

    public File createEmptyTempFile(String fileName) throws IOException {
        log.debug("Starting createEmptyTempFile ("+""+"fileName : "+fileName + " , "+")");
        File tempFile = new File(tempDir, fileName);
        return tempFile;
    }

    public static byte[] getBinaryTestData(int howMuch) {
        log.debug("Starting getBinaryTestData ("+""+"howMuch : "+howMuch + " , "+")");
        byte[] testData = new byte[howMuch];
        for (int i=0; i<testData.length; i++) { testData[i] = (byte)(i % 255); }

        return testData;
    }

    public String toString() {
        log.debug("Starting toString");
        return tempDir.toString();
    }

    /**
     * Create a dummy binary file.  Also useful for tests.
     */
    public File createTempBinaryFile(String fileName, int size) throws IOException {
        log.debug("Starting createTempBinaryFile ("+""+"fileName : "+fileName + " , "+"size : "+size + " , "+")");
        File tempFile = new File(tempDir, fileName);

        DataOutputStream out = new DataOutputStream(new FileOutputStream(tempFile));
        out.write(getBinaryTestData(size));
        out.flush(); out.close();

        return tempFile;
    }

    public boolean isEmpty() {
        log.debug("Starting isEmpty");
        return (listFiles().length==0);
    }

    public void delete() throws IOException {
        log.debug("Starting delete");
        empty();
        tempDir.delete();
    }

    public File getDir() {
        log.debug("Starting getDir");
        return tempDir;
    }

    public boolean containsFileName(String fileName) {
        log.debug("Starting containsFileName ("+""+"fileName : "+fileName + " , "+")");
        String[] files = tempDir.list();
        for (int i=0; i<files.length; i++) {
            // Note - this is case-sensitive
            if (files[i].equals(fileName)) { return true; }
        }
        return false;
    }

    /**
     * This should run when the TempDirectory is garbage collected - it hasn't been tested
     */
    protected void finalize() throws Throwable {
        log.debug("Starting finalize");
//        delete();
    }

}
