/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 **************************************************************************/

package com.day.jcr.vault.fs;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collection;
import java.util.LinkedList;

import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Value;

import org.apache.commons.io.IOUtils;

import com.day.jcr.vault.fs.api.AccessType;
import com.day.jcr.vault.fs.api.Artifact;
import com.day.jcr.vault.fs.api.ArtifactType;
import com.day.jcr.vault.fs.api.DumpContext;
import com.day.jcr.vault.fs.api.ExportArtifact;
import com.day.jcr.vault.fs.api.SerializationType;
import com.day.jcr.vault.fs.api.VaultInputSource;

/**
 * Implements a artifact that is based on a property value.
 *
 */
public class PropertyValueArtifact implements ExportArtifact {
    /**
     * The property of this artifact
     */
    private final Property property;

    /**
     * the path to the property
     */
    private final String path;

    /**
     * Temporary file if the value is detached
     */
    private File tmpFile;

    /**
     * cached content length
     */
    private Long contentLength;

    private final long lastModified;

    /**
     * The value index for multi value properties.
     */
    private final int valueIndex;

    public PropertyValueArtifact(Artifact parent, String relPath, String ext, ArtifactType type,
                                 Property prop, long lastModified)
            throws RepositoryException {
        throw new UnsupportedOperationException("No longer supported. use the org.apache.jackrabbit.vault counterpart.");
    }

    public PropertyValueArtifact(Artifact parent, String relPath, String ext, ArtifactType type,
                                 Property prop, int index, long lastModified)
            throws RepositoryException {
        throw new UnsupportedOperationException("No longer supported. use the org.apache.jackrabbit.vault counterpart.");
    }

    /**
     * Creates a collection of {@link PropertyValueArtifact} from the given
     * property. If the property is multivalued there will be an artifact
     * created for each value with the value index appended to it's name.
     *
     * @param parent parent artifact
     * @param relPath the base name for the artifact(s).
     * @param ext the extension
     * @param type the type for the artifact(s).
     * @param prop the property for the artifact(s).
     * @param lastModified the last modified date.
     *
     * @return a collection of Artifacts.
     * @throws RepositoryException if an error occurs
     */
    public static Collection<PropertyValueArtifact> create(Artifact parent,
                   String relPath, String ext, ArtifactType type, Property prop, long lastModified)
            throws RepositoryException {
        LinkedList<PropertyValueArtifact> list = new LinkedList<PropertyValueArtifact>();
        if (prop.getDefinition().isMultiple()) {
            Value[] values = prop.getValues();
            for (int i=0; i<values.length; i++) {
                StringBuffer n = new StringBuffer(relPath);
                n.append('[').append(i).append(']');
                list.add(new PropertyValueArtifact(parent, n.toString(), ext, type, prop, i, lastModified));
            }
        } else {
            list.add(new PropertyValueArtifact(parent, relPath, ext, type, prop, lastModified));
        }
        return list;
    }

    /**
     * {@inheritDoc}
     */
    public SerializationType getSerializationType() {
        return SerializationType.GENERIC;
    }

    /**
     * {@inheritDoc}
     *
     * @return always {@link AccessType#STREAM}
     */
    public AccessType getPreferredAccess() {
        return AccessType.STREAM;
    }

    /**
     * {@inheritDoc}
     */
    public InputStream getInputStream() throws IOException, RepositoryException {
        return tmpFile == null ?  new PVAInputStream() : new FileInputStream(tmpFile);
    }

    /**
     * Detaches the value from the underlying property value.
     *
     * @throws IOException if an I/O error occurs
     * @throws RepositoryException if a repository error occurs.
     */
    public void detach() throws IOException, RepositoryException {
        if (tmpFile == null) {
            // ensure caching of content type
            getContentType();
            // copy value to temp file
            tmpFile = File.createTempFile("jcrfs", "dat");
            tmpFile.setLastModified(getLastModified());
            tmpFile.deleteOnExit();
            FileOutputStream out = new FileOutputStream(tmpFile);
            InputStream in = getValue().getStream();
            IOUtils.copy(in, out);
            in.close();
            out.close();
        }
    }

    /**
     * {@inheritDoc}
     *
     * @return a input source which systemId is the path of the underlying property
     */
    public VaultInputSource getInputSource() throws IOException, RepositoryException {
        final InputStream in = getInputStream();
        return new VaultInputSource() {

            @Override
            public String getSystemId() {
                return path;
            }

            @Override
            public InputStream getByteStream() {
                return in;
            }


            public long getContentLength() {
                return PropertyValueArtifact.this.getContentLength();
            }

            public long getLastModified() {
                return PropertyValueArtifact.this.getLastModified();
            }
        };
    }

    /**
     * Returns the value either from field or from property.
     *
     * @return the jcr value.
     * @throws RepositoryException if an repository error occurs.
     */
    private Value getValue() throws RepositoryException {
        if (valueIndex < 0) {
            return property.getValue();
        } else {
            Value[] values = property.getValues();
            if (valueIndex >= values.length) {
                throw new RepositoryException("Illegal value index: " + valueIndex);
            }
            return values[valueIndex];
        }
    }

    /**
     * {@inheritDoc}
     */
    public long getContentLength() {
        if (contentLength == null) {
            if (tmpFile == null) {
                contentLength = -1L;
                try {
                    if (valueIndex < 0) {
                        contentLength = property.getLength();
                    } else {
                        long[] lengths = property.getLengths();
                        if (valueIndex < lengths.length) {
                            contentLength = lengths[valueIndex];
                        }
                    }
                } catch (RepositoryException e) {
                    // ignore
                }
            } else {
                contentLength = tmpFile.length();
            }
        }
        return contentLength;
    }

    /**
     * Returns the underlying property
     * @return the underlying property
     */
    public Property getProperty() {
        return property;
    }

    /**
     * {@inheritDoc}
     */
    public long getLastModified() {
        return lastModified;
    }

    /**
     * {@inheritDoc}
     */
    public String getContentType() {
        throw new UnsupportedOperationException("No longer supported. use the org.apache.jackrabbit.vault counterpart.");
    }

    public String getPlatformPath() {
        return null;
    }

    public String getExtension() {
        return null;
    }

    public String getRelativePath() {
        return null;
    }

    public ArtifactType getType() {
        return null;
    }

    public void spool(OutputStream out) throws IOException, RepositoryException {
    }

    public void dump(DumpContext ctx, boolean isLast) {
    }

    /**
     * Internal defered input stream on this property value
     */
    private class PVAInputStream extends InputStream {

        private InputStream stream;

        private boolean closed;

        private void assertOpen() throws IOException {
            if (stream == null) {
                if (closed) {
                    throw new IOException("Stream already closed.");
                }
                try {
                    stream = getValue().getStream();
                } catch (RepositoryException e) {
                    throw new IOException("Error while opening stream: " + e.toString());
                }
            }
        }
        public int read() throws IOException {
            assertOpen();
            return stream.read();
        }

        public int read(byte[] b) throws IOException {
            assertOpen();
            return stream.read(b);
        }

        public int read(byte[] b, int off, int len) throws IOException {
            assertOpen();
            return stream.read(b, off, len);
        }

        public long skip(long n) throws IOException {
            assertOpen();
            return stream.skip(n);
        }

        public int available() throws IOException {
            assertOpen();
            return stream.available();
        }

        public void close() throws IOException {
            try {
                if (stream != null) {
                    stream.close();
                }
            } finally {
                closed = true;
                stream = null;
            }
        }

        public void mark(int readlimit) {
            try {
                assertOpen();
            } catch (IOException e) {
                throw new IllegalStateException(e);
            }
            stream.mark(readlimit);
        }

        public void reset() throws IOException {
            assertOpen();
            stream.reset();
        }

        public boolean markSupported() {
            try {
                assertOpen();
            } catch (IOException e) {
                throw new IllegalStateException(e);
            }
            return stream.markSupported();
        }
    }

}