/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 **************************************************************************/

package com.day.jcr.vault.fs;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import javax.jcr.RepositoryException;

import org.apache.commons.io.output.DeferredFileOutputStream;

import com.day.jcr.vault.fs.api.AccessType;
import com.day.jcr.vault.fs.api.Artifact;
import com.day.jcr.vault.fs.api.ArtifactType;
import com.day.jcr.vault.fs.api.DumpContext;
import com.day.jcr.vault.fs.api.ExportArtifact;
import com.day.jcr.vault.fs.api.SerializationType;
import com.day.jcr.vault.fs.api.VaultInputSource;
import com.day.jcr.vault.fs.io.Serializer;
import com.day.jcr.vault.util.TempFileInputStream;

/**
 * Implements an output artifact that is based on a serializer, i.e. the
 * preferred access method is {@link AccessType#SPOOL}
 *
 */
public class SerializerArtifact implements ExportArtifact {

    /** the serializer that is able to spool the content */
    private final Serializer serializer;

    private final long lastModified;

    /**
     * Constructs a new artifact that is based on a content serializer.
     *
     * @param parent the parent artifact
     * @param name the name of the artifact
     * @param ext the extension of the artifact
     * @param type the type of the artifact
     * @param serializer the serializer to use for the content
     * @param lastModified the last modified date
     *
     * @throws IllegalArgumentException if the type is not suitable.
     */
    public SerializerArtifact(Artifact parent, String name, String ext, ArtifactType type,
                              Serializer serializer, long lastModified) {
        throw new UnsupportedOperationException("No longer supported. use the org.apache.jackrabbit.vault counterpart.");
    }

    /**
     * {@inheritDoc}
     *
     * @return always {@link AccessType#SPOOL}
     */
    public AccessType getPreferredAccess() {
        return AccessType.SPOOL;
    }

    /**
     * {@inheritDoc}
     */
    public SerializationType getSerializationType() {
        return serializer.getType();
    }

    /**
     * {@inheritDoc}
     */
    public void spool(OutputStream out)
            throws IOException, RepositoryException {
        serializer.writeContent(out);
    }

    /**
     * {@inheritDoc}
     */
    public InputStream getInputStream() throws IOException, RepositoryException {
        DeferredFileOutputStream out = new DeferredFileOutputStream(8192, "vlttmp", ".tmp", null);
        spool(out);
        out.close();
        if (out.isInMemory()) {
            return new ByteArrayInputStream(out.getData());
        } else {
            return new TempFileInputStream(out.getFile());
        }
    }

    /**
     * {@inheritDoc}
     */
    public VaultInputSource getInputSource() throws IOException, RepositoryException {
        DeferredFileOutputStream out = new DeferredFileOutputStream(8192, "vlttmp", ".tmp", null);
        spool(out);
        out.close();
        final InputStream in;
        final long size;
        if (out.isInMemory()) {
            in = new ByteArrayInputStream(out.getData());
            size = out.getData().length;
        } else {
            in = new TempFileInputStream(out.getFile());
            size = out.getFile().length();
        }
        return new VaultInputSource() {

            @Override
            public String getSystemId() {
                return SerializerArtifact.this.getRelativePath();
            }

            @Override
            public InputStream getByteStream() {
                return in;
            }


            public long getContentLength() {
                return size;
            }

            public long getLastModified() {
                return lastModified;
            }
        };
    }

    /**
     * {@inheritDoc}
     */
    public String getContentType() {
        throw new UnsupportedOperationException("No longer supported. use the org.apache.jackrabbit.vault counterpart.");
    }

    /**
     * {@inheritDoc}
     */
    public long getContentLength() {
        return -1;
    }

    /**
     * {@inheritDoc}
     */
    public long getLastModified() {
        return lastModified;
    }

    public String getPlatformPath() {
        return null;
    }

    public String getExtension() {
        return null;
    }

    public String getRelativePath() {
        return null;
    }

    public ArtifactType getType() {
        return null;
    }

    public void dump(DumpContext ctx, boolean isLast) {
    }
}