/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 **************************************************************************/

package com.day.jcr.vault.fs;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.DigestOutputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import javax.jcr.RepositoryException;

import com.day.jcr.vault.fs.api.AccessType;
import com.day.jcr.vault.fs.api.Artifact;
import com.day.jcr.vault.fs.api.VaultFile;
import com.day.jcr.vault.util.BinaryCheckOutputStream;
import com.day.jcr.vault.util.LineOutputStream;
import com.day.jcr.vault.util.MD5;

/**
 * <code>FileUtil</code>...
 *
 */
public class VaultFileCopy {

    private final VaultFile remoteFile;

    private final File localFile;

    private final MessageDigest digest;

    private byte[] lineFeed = null;

    private MD5 md5 = null;

    private long length;

    private boolean binary;

    private VaultFileCopy(VaultFile remote, File local, MessageDigest digest, byte[] lineFeed) {
        this.remoteFile = remote;
        this.localFile = local;
        this.digest = digest;
        this.lineFeed = lineFeed;
    }

    public static VaultFileCopy copy(VaultFile remote, File local) throws IOException {
        return copy(remote, local, null);
    }

    public static VaultFileCopy copy(VaultFile remote, File local, byte[] lineFeed) throws IOException {
        MessageDigest md;
        try {
            md = MessageDigest.getInstance("md5");
        } catch (NoSuchAlgorithmException e) {
            throw new IllegalArgumentException(e.toString());
        }
        VaultFileCopy copy = new VaultFileCopy(remote, local, md, lineFeed);
        try {
            copy.run();
        } catch (RepositoryException e) {
            throw new IOException(e.toString());
        }
        return copy;
    }

    private void run() throws IOException, RepositoryException {
        Artifact a = remoteFile.getArtifact();
        if (a.getPreferredAccess() == AccessType.NONE) {
            throw new IOException("Artifact has no content.");
        }
        OutputStream base;
        if (digest == null) {
            base = new FileOutputStream(localFile);
        } else {
            base = new DigestOutputStream(
                    new FileOutputStream(localFile), digest);
        }
        if (lineFeed != null) {
            base = new LineOutputStream(base, lineFeed);
        }
        BinaryCheckOutputStream out = new BinaryCheckOutputStream(base);
        switch (a.getPreferredAccess()) {
            case SPOOL:
                a.spool(out);
                out.close();
                break;
            case STREAM:
                InputStream in = a.getInputStream();
                byte[] buffer = new byte[8192];
                int read;
                while ((read = in.read(buffer)) >= 0) {
                    out.write(buffer, 0, read);
                }
                in.close();
                out.close();
                break;
        }
        binary = out.isBinary();
        length = localFile.length();
        // try to set last modified
        long lastMod = remoteFile.lastModified();
        if (lastMod > 0) {
            localFile.setLastModified(lastMod);
        }
        if (digest != null) {
            md5 = new MD5(digest.digest());
        }
    }

    public MD5 getMd5() {
        return md5;
    }

    public long getLength() {
        return length;
    }

    public boolean isBinary() {
        return binary;
    }
}