/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 **************************************************************************/

package com.day.jcr.vault.fs;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;

import javax.jcr.RepositoryException;

import com.day.jcr.vault.fs.api.AccessType;
import com.day.jcr.vault.fs.api.Artifact;
import com.day.jcr.vault.fs.api.VaultFile;

/**
 * Implements an input stream on a jcr file file. It accesses the artifact of the
 * platform file an wraps either it's input stream or provides one using a tmp
 * file.
 *
 */
public class VaultFileInputStream extends InputStream {

    /**
     * The base input stream
     */
    private final InputStream base;

    /**
     * Temp file for spooling
     */
    private File tmpFile;

    /**
     * Creates a new input stream on the given file. If the file is a
     * directory an IOException is thrown.
     *
     * @param file the file
     * @throws IOException if an I/O error occurs.
     */
    public VaultFileInputStream(VaultFile file) throws IOException {
        Artifact a = file.getArtifact();
        if (a == null || a.getPreferredAccess() == AccessType.NONE) {
            throw new IOException("invalid access.");
        }
        try {
            if (a.getPreferredAccess() == AccessType.STREAM) {
                base = a.getInputStream();
            } else {
                tmpFile = File.createTempFile("vltfs", ".spool");
                FileOutputStream out = new FileOutputStream(tmpFile);
                a.spool(out);
                out.close();
                base = new FileInputStream(tmpFile);
            }
        } catch (RepositoryException e) {
            throw new IOException(e.toString());
        }
    }

    /**
     * {@inheritDoc}
     */
    public int read() throws IOException {
        return base.read();
    }

    /**
     * {@inheritDoc}
     */
    public int read(byte[] b) throws IOException {
        return base.read(b);
    }

    /**
     * {@inheritDoc}
     */
    public int read(byte[] b, int off, int len) throws IOException {
        return base.read(b, off, len);
    }

    /**
     * {@inheritDoc}
     */
    public long skip(long n) throws IOException {
        return base.skip(n);
    }

    /**
     * {@inheritDoc}
     */
    public int available() throws IOException {
        return base.available();
    }

    /**
     * {@inheritDoc}
     */
    public void close() throws IOException {
        base.close();
        if (tmpFile != null) {
            tmpFile.delete();
            tmpFile.deleteOnExit();
            tmpFile = null;
        }
    }

    /**
     * {@inheritDoc}
     */
    public void mark(int readlimit) {
        base.mark(readlimit);
    }

    /**
     * {@inheritDoc}
     */
    public void reset() throws IOException {
        base.reset();
    }

    /**
     * {@inheritDoc}
     */
    public boolean markSupported() {
        return base.markSupported();
    }
}