/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 **************************************************************************/

package com.day.jcr.vault.fs.api;

import java.io.InputStream;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

/**
 * <code>WorkspaceFilter</code>...
 */
public interface WorkspaceFilter extends Dumpable {

    /**
     * Returns a list of path filter sets.
     * @return the list of path filter sets.
     */
    List<PathFilterSet> getFilterSets();

    /**
     * Returns the filter set that covers the respective path
     * @param path the path
     * @return the filter set or <code>null</code>
     */
    PathFilterSet getCoveringFilterSet(String path);

    /**
     * Returns the import mode for the given path.
     * @param path path to check
     * @return the import mode or {@link ImportMode#REPLACE} if the given path
     *         is not covered by this filter.
     */
    ImportMode getImportMode(String path);

    /**
     * Checks if the given path is contained in this workspace filter.
     * It returns <code>true</code> if any of the filter sets contain the path
     * and it's not globally ignored.
     *
     * @param path to check
     * @return <code>true</code> if the given path is included in this filter.
     */
    boolean contains(String path);

    /**
     * Checks if the given path is covered in this workspace filter.
     * It only returns <code>true</code> if at least one of the sets covers
     * the path and is not globally ignored.
     *
     * @param path the pathto check
     * @return <code>true</code> if the given path is covered by this filter.
     */
    boolean covers(String path);

    /**
     * Checks if the given path is an ancestor of any of the filter sets.
     *
     * @param path the item to check
     * @return <code>true</code> if the given item is an ancestor
     */
    boolean isAncestor(String path);

    /**
     * Checks if the given path is globally ignored.
     *
     * @param path the path to check.
     * @return <code>true</code> if the item is globally ignored.
     */
    boolean isGloballyIgnored(String path);

    /**
     * Returns the source xml that constructs this filter
     * @return the source xml
     */
    InputStream getSource();

    /**
     * Returns the source xml that constructs this filter
     * @return the source xml
     */
    String getSourceAsString();

    /**
     * Translates this workspace filter using the given path mapping.
     *
     * @param mapping the path mapping
     * @return a new workspace filter
     * @since 2.4.10
     */
    WorkspaceFilter translate(PathMapping mapping);

    /**
     * Dumps the coverage of this filter against the given node to the listener.
     * @param rootNode root node
     * @param listener listener
     * @throws RepositoryException if an error occurs
     */
    void dumpCoverage(Node rootNode, ProgressTrackerListener listener)
            throws RepositoryException;

    /**
     * Dumps the coverage of this filter using the given session. The traversal starts
     * at the common ancestor of all filter sets. If <code>skipJcrContent</code> is <code>true</code>
     * the jcr:content nodes are excluded from traversal and reporting.
     *
     * @param session session
     * @param listener listener to report progress
     * @param skipJcrContent <code>true</code> to skip jcr:content nodes
     * @throws RepositoryException if an error occurs
     */
    void dumpCoverage(Session session, ProgressTrackerListener listener, boolean skipJcrContent)
            throws RepositoryException;
}