/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 **************************************************************************/

package com.day.jcr.vault.fs.filter;

import javax.jcr.Item;
import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.RepositoryException;

import com.day.jcr.vault.fs.api.DumpContext;

/**
 * Filter that checks the declared type of an item
 *
 */
public class DeclaringTypeItemFilter extends DepthItemFilter {

    /**
     * The node type to check
     */
    private String nodeType;

    /**
     * indicates if only props should be checked
     */
    private boolean propsOnly;

    /**
     * Default constructor.
     */
    public DeclaringTypeItemFilter() {
    }

    /**
     * Creates a new filter for the given node type and flags.
     * @param nodeType the node type name to check
     * @param propsOnly if <code>true</code> only properties are checked
     * @param minDepth the minimal depth
     * @param maxDepth the maximal depth
     */
    public DeclaringTypeItemFilter(String nodeType, boolean propsOnly,
                                   int minDepth, int maxDepth) {
        super(minDepth, maxDepth);
        this.nodeType = nodeType;
        this.propsOnly = propsOnly;
    }

    /**
     * Creates a new filter for the given node type and flags
     * @param nodeType the node type name to check
     * @param propsOnly if <code>true</code> only properties are checked
     */
    public DeclaringTypeItemFilter(String nodeType, boolean propsOnly) {
        this(nodeType, propsOnly, 0, Integer.MAX_VALUE);
    }

    /**
     * Sets the node type to match the declaring one of the item
     * @param nodeType the node type
     */
    public void setNodeType(String nodeType) {
        this.nodeType = nodeType;
    }

    /**
     * Sets the flag that indicates if only properties are to be checked.
     * @param propsOnly if <code>true</code> only properties are checked.
     */
    public void setPropsOnly(String propsOnly) {
        this.propsOnly = Boolean.valueOf(propsOnly);
    }

    /**
     * {@inheritDoc}
     *
     * Matches if the declaring node type of the item is equal to the one
     * specified in this filter. If the item is a node and <code>propsOnly</code>
     * flag is <code>true</code> it returns <code>false</code>.
     */
    public boolean matches(Item item) throws RepositoryException {
        if (item.isNode()) {
            return !propsOnly && ((Node) item).getDefinition().getDeclaringNodeType().getName().equals(nodeType);
        } else {
            return ((Property) item).getDefinition().getDeclaringNodeType().getName().equals(nodeType);
        }
    }

    /**
     * {@inheritDoc}
     */
    public void dump(DumpContext ctx, boolean isLast) {
        super.dump(ctx, isLast);
        ctx.indent(isLast);
        ctx.printf(false, "nodeType: %s", nodeType);
        ctx.printf(true, "propsOnly: %b", propsOnly);
        ctx.outdent();
    }
    
}