/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 **************************************************************************/

package com.day.jcr.vault.fs.filter;

import java.util.regex.Pattern;

import com.day.jcr.vault.fs.api.DumpContext;
import com.day.jcr.vault.fs.api.PathFilter;
import com.day.jcr.vault.fs.api.PathMapping;

/**
 * The default path filter provides hierarchical filtering.
 *
 */
public class DefaultPathFilter implements PathFilter {

    /**
     * the internal regex pattern
     */
    private Pattern regex;

    /**
     * Default constructor
     */
    public DefaultPathFilter() {
    }

    /**
     * Creates a new default path filter
     * @param pattern the pattern
     * @see #setPattern
     */
    public DefaultPathFilter(String pattern) {
        setPattern(pattern);
    }

    /**
     * Sets the regexp pattern for this filter.
     *
     * Examples:
     * <xmp>
     * | Pattern        | Matches
     * | /foo           | exactly "/foo"
     * | /foo.*         | all paths starting with "/foo"
     * | ^.* /foo[^/]*$ | all files starting with "foo"
     * | /foo/[^/]*$    | all direct children of /foo
     * | /foo/.*        | all children of /foo
     * | /foo(/.*)?     | all children of /foo and foo itself
     * </xmp>
     *
     * @param pattern the pattern.
     */
    public void setPattern(String pattern) {
        regex = Pattern.compile(pattern);
    }

    /**
     * Returns the pattern
     * @return the pattern
     */
    public String getPattern() {
        return regex.pattern();
    }

    /**
     * {@inheritDoc}
     */
    public boolean matches(String path) {
        return regex.matcher(path).matches();
    }

    /**
     * {@inheritDoc}
     */
    public boolean isAbsolute() {
        return regex.pattern().startsWith("/");
    }

    /**
     * {@inheritDoc}
     */
    public PathFilter translate(PathMapping mapping) {
        if (mapping == null) {
            return this;
        }
        String pattern = regex.pattern();
        if (!pattern.startsWith("/")) {
            return this;
        }
        return new DefaultPathFilter(mapping.map(pattern));
    }

    /**
     * {@inheritDoc}
     */
    public void dump(DumpContext ctx, boolean isLast) {
        ctx.printf(isLast, "%s:", getClass().getSimpleName());
        ctx.indent(isLast);
        ctx.printf(true, "regex: %s", regex.toString());
        ctx.outdent();
    }
}