/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 **************************************************************************/

package com.day.jcr.vault.fs.filter;

import javax.jcr.Item;
import javax.jcr.RepositoryException;

import com.day.jcr.vault.fs.api.DumpContext;
import com.day.jcr.vault.fs.api.ItemFilter;

/**
 * Implements a filter that filters item according to their (passed) depth.
 *
 */
public class DepthItemFilter implements ItemFilter {

    /**
     * The minimal depth
     */
    private int minDepth = 0;

    /**
     * The maximal depth
     */
    private int maxDepth = Integer.MAX_VALUE;

    /**
     * Default constructor.
     */
    public DepthItemFilter() {
    }

    /**
     * Creates a new depth filter for the given depths.
     * @param minDepth the minimal depth
     * @param maxDepth the maximal depth
     */
    public DepthItemFilter(int minDepth, int maxDepth) {
        this.minDepth = minDepth;
        this.maxDepth = maxDepth;
    }

    /**
     * Sets the minimal depth
     * @param minDepth the minimal depth
     */
    public void setMinDepth(String minDepth) {
        this.minDepth = Integer.decode(minDepth);
    }

    /**
     * Sets the maximal depth
     * @param maxDepth the maximal depth
     */
    public void setMaxDepth(String maxDepth) {
        this.maxDepth = Integer.decode(maxDepth);
    }

    /**
     * {@inheritDoc}
     *
     * Matches if the given depth is greater or equal the minimum depth and
     * less or equal the maximum depth and if the call to {@link #matches(Item)}
     * returns <code>true</code>.
     */
    public boolean matches(Item item, int depth) throws RepositoryException {
        return depth >= minDepth && depth <= maxDepth && matches(item);
    }

    /**
     * Returns <code>true</code>. Subclasses can override to implement something
     * useful that is dependant of the depth.
     * 
     * @param item the item to match
     * @return <code>true</code> if the item matches; <code>false</code> otherwise.
     * @throws RepositoryException if an error occurs.
     */
    public boolean matches(Item item) throws RepositoryException {
        return true;
    }

    /**
     * {@inheritDoc}
     */
    public void dump(DumpContext ctx, boolean isLast) {
        ctx.printf(isLast, "%s:", getClass().getSimpleName());
        ctx.indent(isLast);
        ctx.printf(false, "minDepth: %d", minDepth);
        ctx.printf(true, "maxDepth: %d", maxDepth);
        ctx.outdent();
    }
}