/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 **************************************************************************/

package com.day.jcr.vault.fs.filter;

import javax.jcr.Item;
import javax.jcr.Node;
import javax.jcr.RepositoryException;

import com.day.jcr.vault.fs.api.DumpContext;

/**
 * Filters on the node type of a node.
 *
 */
public class NodeTypeItemFilter extends DepthItemFilter {

    /**
     * the node type to filter on
     */
    private String nodeType;

    /**
     * indicates if super types should be respected
     */
    private boolean respectSupertype;

    /**
     * Default constructor
     */
    public NodeTypeItemFilter() {
    }

    /**
     * Creates a new node type filter.
     * @param nodeType the node type to filter on
     * @param respectSupertype indicates if super type should be respected
     * @param minDepth the minimal depth
     * @param maxDepth the maximal depth
     */
    public NodeTypeItemFilter(String nodeType, boolean respectSupertype,
                              int minDepth, int maxDepth) {
        super(minDepth, maxDepth);
        this.nodeType = nodeType;
        this.respectSupertype = respectSupertype;
    }

    /**
     * Creates a new node type filter.
     * @param nodeType the node type to filter on
     * @param respectSupertype indicates if super type should be respected
     */
    public NodeTypeItemFilter(String nodeType, boolean respectSupertype) {
        this(nodeType, respectSupertype, 0, Integer.MAX_VALUE);
    }

    /**
     * Sets the node type to filter on
     * @param nodeType the node type
     */
    public void setNodeType(String nodeType) {
        this.nodeType = nodeType;
    }

    /**
     * Sets the flag that indicates if super type should be respected.
     * @param respectSupertype if <code>true</code>, super types are respected.
     */
    public void setRespectSupertype(String respectSupertype) {
        this.respectSupertype = Boolean.valueOf(respectSupertype);
    }

    /**
     * {@inheritDoc}
     *
     * Returns <code>true</code> if the item is a node and if the configured
     * node type is equal to the primary type of the node. if super types are
     * respected it also returns <code>true</code> if the items node type
     * extends from the configured node type (Node.isNodeType() check).
     */
    public boolean matches(Item item) throws RepositoryException {
        if (item.isNode()) {
            if (respectSupertype) {
                try {
                    return ((Node) item).isNodeType(nodeType);
                } catch (RepositoryException e) {
                    // ignore
                    return false;
                }
            } else {
                return ((Node) item).getPrimaryNodeType().getName().equals(nodeType);
            }
        }
        return false;
        
    }

    /**
     * {@inheritDoc}
     */
    public void dump(DumpContext ctx, boolean isLast) {
        super.dump(ctx, isLast);
        ctx.indent(isLast);
        ctx.printf(false, "nodeType: %s", nodeType);
        ctx.printf(true, "respectSupertype: %b", respectSupertype);
        ctx.outdent();
    }
    
}