/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 **************************************************************************/

package com.day.jcr.vault.fs.filter;

import java.util.HashSet;
import java.util.Set;

import javax.jcr.Item;
import javax.jcr.Node;
import javax.jcr.RepositoryException;

import com.day.jcr.vault.fs.api.DumpContext;
import com.day.jcr.vault.fs.api.ItemFilter;
import com.day.jcr.vault.util.JcrConstants;

/**
 * The nt file item filter matches all properties that are defined my the
 * nt:file or nt:resource node type. the later only, if the respective nodes
 * name is 'jcr:content'.
 *
 * Additionally the properties 'jcr:encoding' can be configured to be excluded.
 *
 */
public class NtFileItemFilter implements ItemFilter {

    private static final Set<String> fileNames = new HashSet<String>();
    static {
        fileNames.add(JcrConstants.JCR_PRIMARYTYPE);
        fileNames.add(JcrConstants.JCR_MIXINTYPES);
        fileNames.add(JcrConstants.JCR_UUID);
        fileNames.add(JcrConstants.JCR_CREATED);
        fileNames.add(JcrConstants.JCR_CREATED_BY);
    }

    private static final Set<String> resNames = new HashSet<String>();
    static {
        resNames.add(JcrConstants.JCR_ENCODING);
        resNames.add(JcrConstants.JCR_MIMETYPE);
        resNames.add(JcrConstants.JCR_PRIMARYTYPE);
        resNames.add(JcrConstants.JCR_MIXINTYPES);
        resNames.add(JcrConstants.JCR_UUID);
        resNames.add(JcrConstants.JCR_LASTMODIFIED);
        resNames.add(JcrConstants.JCR_DATA);
    }

    /**
     * {@inheritDoc}
     *
     * @return <code>true</code> if the item is a nt:file or nt:resource property
     */
    public boolean matches(Item item, int depth) throws RepositoryException {
        if (item.isNode()) {
            // include nt:file node
            Node node = (Node) item;
            if (depth == 0) {
                return node.isNodeType(JcrConstants.NT_FILE);
            } else if (depth == 1) {
                // include jcr:content
                return item.getName().equals(JcrConstants.JCR_CONTENT);
            } else {
                return false;
            }
        } else {
            if (depth == 1) {
                return fileNames.contains(item.getName());
            } else if (depth == 2 && item.getParent().getName().equals(JcrConstants.JCR_CONTENT)) {
                return resNames.contains(item.getName());
            } else {
                return false;
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    public void dump(DumpContext ctx, boolean isLast) {
        ctx.printf(isLast, "%s:", getClass().getSimpleName());
    }
}