/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 **************************************************************************/

package com.day.jcr.vault.fs.spi;

import java.util.Collection;
import java.util.Map;
import java.util.TreeMap;

import javax.jcr.NamespaceException;
import javax.jcr.RepositoryException;

import org.apache.jackrabbit.spi.Name;
import org.apache.jackrabbit.spi.QNodeTypeDefinition;
import org.apache.jackrabbit.spi.commons.namespace.NamespaceMapping;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 */
public class DefaultNodeTypeSet implements NodeTypeSet {

    /**
     * default logger
     */
    private static final Logger log = LoggerFactory.getLogger(DefaultNodeTypeSet.class);

    private String systemId;

    /**
     * the list of nodetype templates
     */
    private Map<Name, QNodeTypeDefinition> nodeTypes =
            new TreeMap<Name, QNodeTypeDefinition>();

    /**
     * the current namespace mapping
     */
    private NamespaceMapping nsMapping = new NamespaceMapping();

    /**
     * the list of removed nodetype templates
     */
    private Map<Name, QNodeTypeDefinition> removed =
            new TreeMap<Name, QNodeTypeDefinition>();


    public DefaultNodeTypeSet(String systemId) {
        this.systemId = systemId;
    }

    public DefaultNodeTypeSet(NodeTypeSet set) {
        this(set.getSystemId(), set.getNodeTypes().values(), set.getNamespaceMapping());
    }

    public DefaultNodeTypeSet(String systemId,
                              Collection<QNodeTypeDefinition> nodeTypes,
                              NamespaceMapping nsMapping) {
        this.systemId = systemId;
        for (QNodeTypeDefinition t: nodeTypes) {
            this.nodeTypes.put(t.getName(), t);
        }
        this.nsMapping = nsMapping;
    }

    public void add(NodeTypeSet set) {
        for (QNodeTypeDefinition tpl: set.getNodeTypes().values()) {
            log.debug("adding {}", tpl.getName());
            nodeTypes.put(tpl.getName(), tpl);
        }
        add(set.getNamespaceMapping());
    }

    public void add(Collection<QNodeTypeDefinition> set, NamespaceMapping nsMapping) {
        for (QNodeTypeDefinition tpl: set) {
            log.debug("adding {}", tpl.getName());
            nodeTypes.put(tpl.getName(), tpl);
        }
        add(nsMapping);
    }

    private void add(NamespaceMapping mapping) {
        for (Object o : mapping.getPrefixToURIMapping().keySet()) {
            try {
                String pfx = (String) o;
                String uri = mapping.getURI(pfx);
                nsMapping.setMapping(pfx, uri);
            } catch (NamespaceException e) {
                throw new IllegalStateException("Error while transfering mappings.", e);
            }
        }
    }

    public QNodeTypeDefinition remove(Name name)
            throws RepositoryException {
        QNodeTypeDefinition tpl = nodeTypes.remove(name);
        if (tpl != null) {
            removed.put(tpl.getName(), tpl);
            log.debug("removing registered {}", tpl.getName());
        }
        return tpl;
    }

    public Map<Name, QNodeTypeDefinition> getNodeTypes() {
        return nodeTypes;
    }

    public NamespaceMapping getNamespaceMapping() {
        return nsMapping;
    }

    public Map<Name, QNodeTypeDefinition> getRemoved() {
        return removed;
    }

    public String getSystemId() {
        return systemId;
    }

    public void setSystemId(String systemId) {
        this.systemId = systemId;
    }
}