/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 **************************************************************************/

package com.day.jcr.vault.packaging;

import java.util.ArrayList;
import java.util.List;

/**
 * Implements a package dependency reference.
 *
 * @since 2.0
 * @deprecated since 2.5.0. Use the org.apache.jackrabbit.vault API instead.
 */
@Deprecated
public class Dependency {

    /**
     * An empty dependency array
     */
    public static final Dependency[] EMPTY = new Dependency[0];

    /**
     * group id of the dependency
     */
    private final String groupId;

    /**
     * name of the dependency
     */
    private final String name;

    /**
     * Version range of the dependency
     */
    private final VersionRange range;

    /**
     * internal string representation
     */
    private final String str;

    /**
     * Creates a new dependency reference.
     * @param groupId group id
     * @param name name
     * @param range version range
     */
    public Dependency(String groupId, String name, VersionRange range) {
        if (groupId.startsWith(PackageId.ETC_PACKAGES_PREFIX)) {
            groupId = groupId.substring(PackageId.ETC_PACKAGES_PREFIX.length());
        }
        this.groupId = groupId;
        this.name = name;
        this.range = range == null ? VersionRange.INFINITE : range;
        StringBuilder b = new StringBuilder();
        if (groupId.length() > 0 || !VersionRange.INFINITE.equals(this.range)) {
            b.append(groupId);
            b.append(":");
        }
        b.append(name);
        if (!VersionRange.INFINITE.equals(this.range)) {
            b.append(":");
            b.append(range);
        }
        this.str = b.toString();
    }

    /**
     * Creates a new dependency to the specified package id
     * @param id package id.
     */
    public Dependency(PackageId id) {
        this(id.getGroup(), id.getName(), new VersionRange(id.getVersion()));
    }

    /**
     * Returns the group id of the dependency
     * @return the group id
     * @deprecated since 2.4. use {@link #getGroup()} instead.
     */
    public String getGroupId() {
        return groupId;
    }

    /**
     * Returns the group of the dependency
     * @return the group id
     * @since 2.4
     */
    public String getGroup() {
        return groupId;
    }

    /**
     * Returns the name of the dependency
     * @return the name
     */
    public String getName() {
        return name;
    }

    /**
     * Returns the version range
     * @return the version range
     */
    public VersionRange getRange() {
        return range;
    }

    /**
     * Returns the installation path of this dependency
     * @return the installation path;
     */
    public String getPath() {
        StringBuilder b = new StringBuilder();
        if (groupId.length() > 0) {
            if (groupId.charAt(0) != '/') {
                b.append(PackageId.ETC_PACKAGES_PREFIX);
            }
            b.append(groupId);
            b.append("/");
        } else {
            b.append(PackageId.ETC_PACKAGES_PREFIX);
        }
        b.append(name);
        return b.toString();
    }

    /**
     * Checks if the given package id matches this dependency specification.
     * @param id the package id
     * @return <code>true</code> if matches
     */
    public boolean matches(PackageId id) {
        return groupId.equals(id.getGroup())
                && name.equals(id.getName())
                && range.isInRange(id.getVersion());
    }

    /**
     * Returns a dependency from a string. if the given id is null or an
     * empty string, <code>null</code> is returned.
     * @param str the string
     * @return the dependency
     */
    public static Dependency fromString(String str) {
        if (str == null || str.length() == 0) {
            return null;
        }
        String[] segs = str.split(":");
        String name;
        String groupId = "";
        String range = null;
        if (segs.length == 1) {
            name = segs[0];
            // be backward compatible, respect group in name
            int idx = name.lastIndexOf('/');
            if (idx >= 0) {
                groupId = name.substring(0, idx);
                name = name.substring(idx + 1);
            }
        } else if (segs.length == 2) {
            groupId = segs[0];
            name = segs[1];
            boolean isVersion = true;
            if (name.length() > 0) {
                char c = name.charAt(0);
                isVersion = Character.isDigit(c) || c == '[' || c == ')';
            }
            // be backward compatible, respect group in name
            int idx = name.lastIndexOf('/');
            if (idx >= 0 && groupId.length() == 0) {
                groupId = name.substring(0, idx);
                name = name.substring(idx + 1);
            } else if ((idx = groupId.lastIndexOf('/')) >=0 && isVersion) {
                groupId = segs[0].substring(0, idx);
                name = segs[0].substring(idx + 1);
                range = segs[1];
            }
        } else {
            groupId = segs[0];
            name = segs[1];
            range = segs[2];
        }
        return new Dependency(groupId, name, range == null ? null : VersionRange.fromString(range));
    }

    /**
     * Parses a string serialization of dependency references generated by
     * {@link #toString(Dependency...)})}
     *
     * @param str serialized string
     * @return array of dependency references
     */
    public static Dependency[] parse(String str) {
        List<Dependency> deps = new ArrayList<Dependency>();
        boolean inRange = false;
        int start = 0;
        boolean wasSeg = false;
        for (int i=0; i<str.length(); i++) {
            char c = str.charAt(i);
            if (c == ',') {
                if (!inRange) {
                    deps.add(Dependency.fromString(str.substring(start, i)));
                    start = i + 1;
                }
            } else if (c == '[' || c == '(') {
                if (wasSeg) {
                    inRange = true;
                }
            } else if (c == ']' || c == ')') {
                inRange = false;
            }
            wasSeg = c == ':';
        }
        if (start < str.length()) {
            deps.add(Dependency.fromString(str.substring(start)));
        }
        return deps.toArray(new Dependency[deps.size()]);
    }
    
    /**
     * Returns dependencies from the given strings.
     * @param str the strings
     * @return the dependencies
     */
    public static Dependency[] fromString(String ... str) {
        Dependency[] ret = new Dependency[str.length];
        for (int i=0; i<str.length;i++) {
            ret[i] = Dependency.fromString(str[i]);
        }
        return ret;
    }

    /**
     * Returns a string representation from given dependencies.
     * @param deps the dependencies
     * @return the strings
     */
    public static String toString(Dependency ... deps) {
        String delim = "";
        StringBuilder b = new StringBuilder();
        for (Dependency dep: deps) {
            b.append(delim).append(dep);
            delim=",";
        }
        return b.toString();
    }

    @Override
    public String toString() {
        return str;
    }

    @Override
    public boolean equals(Object obj) {
        return this == obj ||
                obj instanceof Dependency && str.equals(obj.toString());
    }

    @Override
    public int hashCode() {
        return str.hashCode();
    }

    public static org.apache.jackrabbit.vault.packaging.Dependency toJrVltDependency(Dependency dep) {
        return dep == null
                ? null
                : org.apache.jackrabbit.vault.packaging.Dependency.fromString(dep.toString());
    }
    public static org.apache.jackrabbit.vault.packaging.Dependency[] toJrVltDependencies(Dependency[] deps) {
        if (deps == null) {
            return null;
        }
        org.apache.jackrabbit.vault.packaging.Dependency[] ret = new org.apache.jackrabbit.vault.packaging.Dependency[deps.length];
        for (int i=0; i<deps.length; i++) {
            ret[i] = toJrVltDependency(deps[i]);
        }
        return ret;
    }
    public static Dependency[] create(org.apache.jackrabbit.vault.packaging.Dependency[] deps) {
        if (deps == null) {
            return null;
        }
        Dependency[] ret = new Dependency[deps.length];
        for (int i=0; i<deps.length; i++) {
            ret[i] = Dependency.create(deps[i]);
        }
        return ret;
    }

    public static Dependency create(org.apache.jackrabbit.vault.packaging.Dependency dep) {
        return dep == null ? null : Dependency.fromString(dep.toString());
    }
 }