/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 **************************************************************************/

package com.day.jcr.vault.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

import org.apache.commons.io.FileUtils;

import com.day.jcr.vault.fs.api.VaultInputSource;

/**
 * Implements a input source that is based on a {@link File}. The path of the
 * file is used as systemId.
 * <p/>
 * Currently only {@link #getByteStream()} is implemented.
 *
 */
public class FileInputSource extends VaultInputSource {

    /**
     * the file
     */
    private final File file;

    /**
     * possible line feed
     */
    private byte[] lineSeparator;

    /**
     * Creates a new input source that is based on a file.
     * @param file the file.
     */
    public FileInputSource(File file) {
        super(file.getPath());
        this.file = file;
    }

    /**
     * Sets the linefeed to use. If this is not <code>null</code> the output
     * stream of the file is wrapped by a {@link LineInputStream} with that
     * given line feed
     * @param lineSeparator the linefeed for text
     */
    public void setLineSeparator(byte[] lineSeparator) {
        this.lineSeparator = lineSeparator;
    }

    /**
     * {@inheritDoc}
     *
     * @return a {@link FileInputStream} on the internal file.
     */
    public InputStream getByteStream() {
        try {
            if (lineSeparator != null) {
                return new LineInputStream(new FileInputStream(file), lineSeparator);
            } else {
                return FileUtils.openInputStream(file);
            }
        } catch (IOException e) {
            return null;
        }
    }

    /**
     * Returns the content length of the underlying file.
     * @return the content length of the underlying file.
     */
    public long getContentLength() {
        return file.length();
    }

    /**
     * Returns the last modified date of the underlying file.
     * @return the last modified date of the underlying file.
     */
    public long getLastModified() {
        return file.lastModified();
    }

    /**
     * deletes the underlying file
     */
    public void discard() {
        file.delete();
        file.deleteOnExit();
    }
}